<div class="toast-container" x-data="toastNotifications()" x-init="init()">
    <template x-for="toast in toasts" :key="toast.id">
        <div x-show="toast.show"
             x-transition:enter="transform ease-out duration-500"
             x-transition:enter-start="translate-x-full opacity-0"
             x-transition:enter-end="translate-x-0 opacity-100"
             x-transition:leave="transition ease-in duration-300"
             x-transition:leave-start="opacity-100 translate-x-0"
             x-transition:leave-end="opacity-0 translate-x-full"
             class="toast-notification" :class="toast.type">
            
            <!-- Progress Bar -->
            <div class="toast-progress" :class="toast.type"></div>
            
            <!-- Icon -->
            <div class="toast-icon" :class="toast.type">
                <i class="fas" :class="{
                    'fa-check-circle': toast.type === 'success',
                    'fa-exclamation-triangle': toast.type === 'warning',
                    'fa-times-circle': toast.type === 'error',
                    'fa-info-circle': toast.type === 'info'
                }"></i>
            </div>

            <!-- Content -->
            <div class="toast-content">
                <div class="toast-title" x-text="toast.title"></div>
                <div class="toast-message" x-text="toast.message"></div>
            </div>

            <!-- Close Button -->
            <button @click="removeToast(toast.id)" class="toast-close" title="ปิด">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </template>
</div>

<script>
function toastNotifications() {
    return {
        toasts: [],
        nextId: 1,

        init() {
            // Listen for custom events
            window.addEventListener('show-toast', (event) => {
                this.addToast(event.detail);
            });

            // Poll for new notifications
            this.pollNotifications();
            setInterval(() => this.pollNotifications(), 10000);
        },

        addToast(data) {
            const toast = {
                id: this.nextId++,
                title: data.title || 'แจ้งเตือน',
                message: data.message || '',
                type: data.type || 'info',
                show: true
            };

            this.toasts.push(toast);
            this.playSound(data.type);

            // Auto remove after 5 seconds
            setTimeout(() => {
                this.removeToast(toast.id);
            }, 5000);
        },

        playSound(type) {
            try {
                // เพิ่ม bell animation
                const bellIcon = document.querySelector('.bell-button .fa-bell');
                if (bellIcon) {
                    bellIcon.style.animation = 'bellShake 0.6s ease-in-out';
                    setTimeout(() => {
                        bellIcon.style.animation = '';
                    }, 600);
                }
                
                // ใช้ Web Audio API ก่อน
                if (window.notificationSound) {
                    window.notificationSound.play(type);
                    return;
                }
                
                // fallback ใช้ HTML5 Audio
                const sounds = {
                    success: '/sounds/success.mp3',
                    warning: '/sounds/warning.mp3',
                    error: '/sounds/error.mp3',
                    info: '/sounds/info.mp3'
                };
                
                const soundUrl = sounds[type] || '/sounds/notification.mp3';
                const audio = new Audio(soundUrl);
                audio.volume = 0.3;
                audio.play().catch(() => {});
            } catch (e) {}
        },

        removeToast(id) {
            const index = this.toasts.findIndex(toast => toast.id === id);
            if (index > -1) {
                this.toasts[index].show = false;
                setTimeout(() => {
                    this.toasts.splice(index, 1);
                }, 300);
            }
        },

        async pollNotifications() {
            try {
                const response = await fetch('/api/notifications/recent');
                const notifications = await response.json();
                
                // Show only new notifications (created in last 30 seconds)
                const thirtySecondsAgo = new Date(Date.now() - 30000);
                notifications.forEach(notification => {
                    const createdAt = new Date(notification.created_at);
                    if (createdAt > thirtySecondsAgo && !notification.read_at) {
                        this.addToast({
                            title: notification.title,
                            message: notification.message,
                            type: notification.type
                        });
                    }
                });
            } catch (error) {
                console.error('Error polling notifications:', error);
            }
        }
    }
}

// Global function to show toast
window.showToast = function(title, message, type = 'info') {
    window.dispatchEvent(new CustomEvent('show-toast', {
        detail: { title, message, type }
    }));
};
</script><?php /**PATH C:\xampp\htdocs\crs\resources\views/components/toast-notifications.blade.php ENDPATH**/ ?>