@extends('layouts.app')

@section('title', 'บัญชีการเงิน')
@section('page-title', 'บัญชีการเงิน')
@section('mobile-title', 'รายรับ-รายจ่าย')

@section('content')
<!-- Welcome Card (Mobile Only) -->
<div class="welcome-card">
    <div class="welcome-content">
        <div class="welcome-text">
            <div class="welcome-greeting">บัญชีการเงิน</div>
            <div class="welcome-name">{{ Auth::user()->name }}</div>
            <div class="welcome-subtitle">จัดการรายรับและรายจ่ายของคุณ</div>
        </div>
        <div class="welcome-actions">
            <button onclick="exportRecords()" class="export-btn-mobile">
                <i class="fas fa-download"></i>
            </button>
            <div class="welcome-icon">
                <i class="fas fa-wallet"></i>
            </div>
        </div>
    </div>
</div>

<div class="records-container">

    <!-- Stats Grid -->
    <div class="stats-grid">
        <div class="stat-card income">
            <div class="stat-icon">
                <i class="fas fa-arrow-up"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">฿{{ number_format($totalIncome ?? 0, 0) }}</div>
                <div class="stat-label">รายรับทั้งหมด</div>
                <div class="stat-detail">
                    <span class="detail-badge income">
                        <i class="fas fa-chart-line"></i>
                        เดือนนี้
                    </span>
                </div>
            </div>
        </div>

        <div class="stat-card expense">
            <div class="stat-icon">
                <i class="fas fa-arrow-down"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">฿{{ number_format($totalExpense ?? 0, 0) }}</div>
                <div class="stat-label">รายจ่ายทั้งหมด</div>
                <div class="stat-detail">
                    <span class="detail-badge expense">
                        <i class="fas fa-chart-line"></i>
                        เดือนนี้
                    </span>
                </div>
            </div>
        </div>

        <div class="stat-card balance">
            <div class="stat-icon">
                <i class="fas fa-{{ ($netAmount ?? 0) >= 0 ? 'piggy-bank' : 'exclamation-triangle' }}"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ ($netAmount ?? 0) >= 0 ? '+' : '-' }}฿{{ number_format(abs($netAmount ?? 0), 0) }}</div>
                <div class="stat-label">ยอดคงเหลือ</div>
                <div class="stat-detail">
                    <span class="detail-badge balance">
                        <i class="fas fa-{{ ($netAmount ?? 0) >= 0 ? 'check-circle' : 'times-circle' }}"></i>
                        {{ ($netAmount ?? 0) >= 0 ? 'เงินเหลือ' : 'ติดลบ' }}
                    </span>
                </div>
            </div>
        </div>

        <div class="stat-card savings">
            <div class="stat-icon">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ ($totalIncome ?? 0) > 0 ? round((($netAmount ?? 0) / $totalIncome) * 100, 1) : 0 }}%</div>
                <div class="stat-label">อัตราการออม</div>
                <div class="stat-detail">
                    <span class="detail-badge savings">
                        <i class="fas fa-target"></i>
                        เป้า 20%
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Filter Section -->
    <div class="filter-section">
        <div class="filter-header">
            <div class="filter-title">
                <i class="fas fa-filter"></i>
                <span>ค้นหาและกรอง</span>
            </div>
            <button class="filter-toggle" onclick="toggleFilters()">
                <i class="fas fa-chevron-down"></i>
            </button>
        </div>
        <div class="filter-content" id="filterContent">
            <form method="GET" action="{{ route('finance.records.index') }}" class="filter-form">
                <div class="filter-row">
                    <div class="filter-field">
                        <label>ประเภท</label>
                        <select name="type" class="form-select">
                            <option value="">ทั้งหมด</option>
                            <option value="income" {{ request('type') === 'income' ? 'selected' : '' }}>💰 รายรับ</option>
                            <option value="expense" {{ request('type') === 'expense' ? 'selected' : '' }}>💸 รายจ่าย</option>
                        </select>
                    </div>
                    
                    <div class="filter-field">
                        <label>หมวดหมู่</label>
                        <select name="category" class="form-select">
                            <option value="">ทุกหมวดหมู่</option>
                            @if(isset($categories['income']) && count($categories['income']) > 0)
                                <optgroup label="รายรับ">
                                    @foreach($categories['income'] as $cat)
                                        <option value="{{ $cat->slug }}" {{ request('category') == $cat->slug ? 'selected' : '' }}>
                                            {{ $cat->name }}
                                        </option>
                                    @endforeach
                                </optgroup>
                            @endif
                            @if(isset($categories['expense']) && count($categories['expense']) > 0)
                                <optgroup label="รายจ่าย">
                                    @foreach($categories['expense'] as $cat)
                                        <option value="{{ $cat->slug }}" {{ request('category') == $cat->slug ? 'selected' : '' }}>
                                            {{ $cat->name }}
                                        </option>
                                    @endforeach
                                </optgroup>
                            @endif
                        </select>
                    </div>
                    
                    <div class="filter-actions">
                        <button type="submit" class="btn-apply">นำไปใช้</button>
                        @if(request()->hasAny(['category', 'type', 'search']))
                            <a href="{{ route('finance.records.index') }}" class="btn-clear">ล้าง</a>
                        @endif
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Search Box -->
    <div class="search-container">
        <form method="GET" action="{{ route('finance.records.index') }}" class="search-form">
            <input type="hidden" name="type" value="{{ request('type') }}">
            <input type="hidden" name="category" value="{{ request('category') }}">
            
            <div class="search-input-wrapper">
                <i class="fas fa-search"></i>
                <input type="text" name="search" placeholder="ค้นหารายการ..." value="{{ request('search') }}">
                <button type="submit" class="search-submit">ค้นหา</button>
            </div>
        </form>
    </div>

    <!-- Records List -->
    @if($records->count() > 0)
        <div class="records-card">
            <div class="records-header">
                <div class="records-title">
                    <i class="fas fa-history"></i>
                    รายการทั้งหมด
                </div>
                <div class="records-count">{{ $records->total() }} รายการ</div>
            </div>
            
            <div class="records-list">
                @foreach($records as $record)
                <div class="record-item">
                    <div class="record-icon">
                        <i class="fas {{ $record->type === 'income' ? 'fa-arrow-up' : 'fa-arrow-down' }}"></i>
                    </div>
                    <div class="record-content">
                        <div class="record-title">
                            <i class="fas {{ $record->categoryInfo->icon ?? 'fa-circle' }}" 
                               style="color: {{ $record->categoryInfo->color ?? '#6B7280' }}"></i>
                            {{ $record->categoryInfo->name ?? $record->category }}
                        </div>
                        <div class="record-meta">
                            <span class="record-date">{{ $record->record_date->format('d M Y') }}</span>
                            @if($record->source === 'delivery')
                                <span class="source-tag delivery">
                                    <i class="fas fa-robot"></i> จากระบบ
                                </span>
                            @endif
                        </div>
                        @if($record->note)
                            <div class="record-note">{{ $record->note }}</div>
                        @endif
                    </div>
                    <div class="record-amount {{ $record->type }}">
                        {{ $record->type === 'income' ? '+' : '-' }}฿{{ number_format($record->amount, 0) }}
                    </div>
                    <div class="record-actions">
                        @if($record->source === 'manual')
                            <button onclick="editRecord({{ $record->id }})" class="action-btn edit" title="แก้ไข">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button onclick="deleteRecord({{ $record->id }})" class="action-btn delete" title="ลบ">
                                <i class="fas fa-trash"></i>
                            </button>
                        @else
                            <button onclick="duplicateRecord({{ $record->id }})" class="action-btn duplicate" title="คัดลอก">
                                <i class="fas fa-copy"></i>
                            </button>
                        @endif
                    </div>
                </div>
                @endforeach
            </div>
            
            @if ($records->hasPages())
                <div class="pagination-container">
                    {{ $records->links() }}
                </div>
            @endif
        </div>
    @else
        <div class="empty-records">
            <i class="fas fa-clipboard-list"></i>
            <p>ยังไม่มีรายการล่าสุด</p>
            <a href="{{ route('finance.records.create', ['type' => 'income']) }}" class="add-record-btn">
                <i class="fas fa-plus"></i>
                เริ่มบันทึก
            </a>
        </div>
    @endif
</div>

<!-- Quick Actions -->
<div class="quick-actions-card">
    <div class="actions-header">
        <div class="actions-title">การดำเนินการด่วน</div>
    </div>
    <div class="actions-grid">
        <a href="{{ route('finance.records.create', ['type' => 'income']) }}" class="action-item primary">
            <div class="action-icon">
                <i class="fas fa-plus"></i>
            </div>
            <span class="action-label">เพิ่มรายรับ</span>
        </a>
        <a href="{{ route('finance.records.create', ['type' => 'expense']) }}" class="action-item secondary">
            <div class="action-icon">
                <i class="fas fa-minus"></i>
            </div>
            <span class="action-label">เพิ่มรายจ่าย</span>
        </a>
        <a href="{{ route('finance.reports.index') }}" class="action-item tertiary">
            <div class="action-icon">
                <i class="fas fa-chart-bar"></i>
            </div>
            <span class="action-label">รายงาน</span>
        </a>
        <a href="{{ route('finance.budgets.index') }}" class="action-item quaternary">
            <div class="action-icon">
                <i class="fas fa-calculator"></i>
            </div>
            <span class="action-label">งบประมาณ</span>
        </a>
    </div>
</div>
</div>

@push('styles')
<style>
/* Welcome Card */
.welcome-card {
    background: var(--gradient);
    border-radius: 20px;
    padding: 20px;
    margin-bottom: 20px;
    color: white;
}

.welcome-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.welcome-greeting {
    font-size: 14px;
    opacity: 0.9;
    margin-bottom: 4px;
}

.welcome-name {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 4px;
}

.welcome-subtitle {
    font-size: 14px;
    opacity: 0.9;
}

.welcome-actions {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.export-btn-mobile {
    background: rgba(255, 255, 255, 0.2);
    border: 1px solid rgba(255, 255, 255, 0.3);
    color: white;
    padding: 0.5rem 0.75rem;
    border-radius: 8px;
    font-size: 0.8rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    backdrop-filter: blur(10px);
    min-width: 50px;
}

.export-btn-mobile:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: scale(1.05);
}

.welcome-icon {
    width: 60px;
    height: 60px;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
}

/* Container */
.records-container {
    padding: 0;
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
    margin-bottom: 20px;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 16px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
}

.stat-card.income::before { background: #10B981; }
.stat-card.expense::before { background: #EF4444; }
.stat-card.balance::before { background: #3B82F6; }
.stat-card.savings::before { background: #8B5CF6; }

.stat-icon {
    width: 44px;
    height: 44px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 12px;
    font-size: 20px;
}

.stat-card.income .stat-icon {
    background: rgba(16,185,129,0.1);
    color: #10B981;
}

.stat-card.expense .stat-icon {
    background: rgba(239,68,68,0.1);
    color: #EF4444;
}

.stat-card.balance .stat-icon {
    background: rgba(59,130,246,0.1);
    color: #3B82F6;
}

.stat-card.savings .stat-icon {
    background: rgba(139,92,246,0.1);
    color: #8B5CF6;
}

.stat-value {
    font-size: 20px;
    font-weight: 700;
    color: var(--gray-900);
    margin-bottom: 4px;
}

.stat-label {
    font-size: 12px;
    color: var(--gray-500);
    font-weight: 500;
    margin-bottom: 8px;
}

.stat-detail {
    display: flex;
    align-items: center;
}

.detail-badge {
    background: var(--gray-100);
    color: var(--gray-600);
    padding: 2px 6px;
    border-radius: 6px;
    font-size: 10px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
}

.detail-badge.income {
    background: rgba(16,185,129,0.1);
    color: #10B981;
}

.detail-badge.expense {
    background: rgba(239,68,68,0.1);
    color: #EF4444;
}

.detail-badge.balance {
    background: rgba(59,130,246,0.1);
    color: #3B82F6;
}

.detail-badge.savings {
    background: rgba(139,92,246,0.1);
    color: #8B5CF6;
}

/* Filter Section */
.filter-section {
    background: white;
    border-radius: 16px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
    overflow: hidden;
}

.filter-header {
    background: var(--gradient-soft);
    padding: 16px 20px;
    border-bottom: 1px solid rgba(255,107,53,0.1);
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.filter-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary);
    display: flex;
    align-items: center;
    gap: 8px;
}

.filter-toggle {
    background: none;
    border: none;
    color: var(--primary);
    font-size: 16px;
    cursor: pointer;
    padding: 8px;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.filter-toggle:hover {
    background: rgba(255, 107, 53, 0.1);
}

.filter-content {
    padding: 20px;
    display: none;
}

.filter-content.show {
    display: block;
}

.filter-row {
    display: grid;
    grid-template-columns: 1fr 1fr auto;
    gap: 1rem;
    align-items: end;
}

.filter-field {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.filter-field label {
    font-size: 0.9rem;
    font-weight: 600;
    color: #374151;
}

.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 1px solid #D1D5DB;
    border-radius: 12px;
    font-size: 0.9rem;
    background: white;
    transition: all 0.2s ease;
}

.form-select:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.filter-actions {
    display: flex;
    gap: 0.5rem;
}

.btn-apply {
    padding: 0.75rem 1.25rem;
    background: #FF6B35;
    color: white;
    border: none;
    border-radius: 12px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-apply:hover {
    background: #E85D2F;
    transform: translateY(-1px);
}

.btn-clear {
    padding: 0.75rem 1.25rem;
    background: #F3F4F6;
    color: #6B7280;
    border: none;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-clear:hover {
    background: #E5E7EB;
    color: #374151;
}

/* Search Container */
.search-container {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    box-shadow: 0 2px 12px rgba(0, 0, 0, 0.06);
}

.search-input-wrapper {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 0.75rem 1rem;
    border: 1px solid #D1D5DB;
    border-radius: 12px;
    background: white;
}

.search-input-wrapper i {
    color: #9CA3AF;
}

.search-input-wrapper input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 0.9rem;
}

.search-submit {
    padding: 0.5rem 1rem;
    background: #FF6B35;
    color: white;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
}

.search-submit:hover {
    background: #E85D2F;
}

/* Records */
.records-card {
    background: white;
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
}

.records-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 16px;
}

.records-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--gray-900);
    display: flex;
    align-items: center;
    gap: 8px;
}

.records-title i {
    color: var(--primary);
}

.records-count {
    font-size: 14px;
    color: var(--gray-500);
    font-weight: 500;
}

.record-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 0;
    border-bottom: 1px solid var(--gray-100);
}

.record-item:last-child {
    border-bottom: none;
}

.record-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--gradient-soft);
    color: var(--primary);
    font-size: 16px;
    flex-shrink: 0;
}

.record-content {
    flex: 1;
}

.record-title {
    font-size: 14px;
    font-weight: 500;
    color: var(--gray-900);
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 6px;
}

.record-meta {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 12px;
    color: var(--gray-500);
}

.record-date {
    background: var(--gray-100);
    padding: 2px 6px;
    border-radius: 4px;
}

.source-tag {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.2rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
}

.source-tag.delivery {
    background: #E0F2FE;
    color: #0284C7;
}

.record-note {
    font-size: 12px;
    color: var(--gray-400);
    margin-top: 2px;
}

.record-amount {
    font-size: 14px;
    font-weight: 600;
    color: var(--primary);
}

.record-amount.income {
    color: #10B981;
}

.record-amount.expense {
    color: #EF4444;
}

.record-actions {
    display: flex;
    gap: 0.5rem;
}

.action-btn {
    width: 32px;
    height: 32px;
    border-radius: 8px;
    border: none;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;
}

.action-btn.edit {
    background: #EFF6FF;
    color: #2563EB;
}

.action-btn.edit:hover {
    background: #DBEAFE;
    transform: scale(1.05);
}

.action-btn.delete {
    background: #FEF2F2;
    color: #DC2626;
}

.action-btn.delete:hover {
    background: #FEE2E2;
    transform: scale(1.05);
}

.action-btn.duplicate {
    background: #FFFBEB;
    color: #D97706;
}

.action-btn.duplicate:hover {
    background: #FEF3C7;
    transform: scale(1.05);
}

.empty-records {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 40px 20px;
    text-align: center;
    color: var(--gray-400);
}

.empty-records i {
    font-size: 48px;
    margin-bottom: 12px;
}

.empty-records p {
    margin-bottom: 16px;
    color: var(--gray-500);
}

.add-record-btn {
    background: var(--gradient);
    color: white;
    padding: 8px 16px;
    border-radius: 12px;
    text-decoration: none;
    font-size: 14px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 6px;
}

/* Quick Actions */
.quick-actions-card {
    background: white;
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 100px;
    box-shadow: 0 2px 12px rgba(255,107,53,0.06);
    border: 1px solid rgba(255,107,53,0.05);
}

.actions-header {
    margin-bottom: 16px;
}

.actions-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--gray-900);
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 12px;
}

.action-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    padding: 16px 8px;
    border-radius: 12px;
    text-decoration: none;
    transition: all 0.3s;
    background: var(--gray-50);
}

.action-item:active {
    transform: scale(0.95);
}

.action-icon {
    width: 44px;
    height: 44px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 8px;
    font-size: 18px;
}

.action-item.primary .action-icon {
    background: var(--gradient-soft);
    color: var(--primary);
}

.action-item.secondary .action-icon {
    background: rgba(239,68,68,0.1);
    color: #EF4444;
}

.action-item.tertiary .action-icon {
    background: rgba(139,92,246,0.1);
    color: #8B5CF6;
}

.action-item.quaternary .action-icon {
    background: rgba(16,185,129,0.1);
    color: #10B981;
}

.action-label {
    font-size: 11px;
    font-weight: 500;
    color: var(--gray-700);
    text-align: center;
}

/* Desktop Styles */
@media (min-width: 769px) {
    .welcome-card {
        display: none;
    }
    
    .stats-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 20px;
    }
    
    .actions-grid {
        grid-template-columns: repeat(6, 1fr);
    }
}

@media (max-width: 380px) {
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .actions-grid {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .filter-row {
        grid-template-columns: 1fr;
    }
}

/* Filter Section */
.filter-section {
    background: white;
    border-radius: 20px;
    margin-bottom: 24px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    overflow: hidden;
}

.filter-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px 24px;
    background: var(--gray-50);
    border-bottom: 1px solid var(--gray-200);
}

.filter-title {
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: var(--gray-900);
    font-size: 16px;
}

.filter-title i {
    color: var(--primary);
}

.filter-toggle {
    background: none;
    border: none;
    color: var(--primary);
    font-size: 16px;
    cursor: pointer;
    padding: 8px;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.filter-toggle:hover {
    background: rgba(255, 107, 53, 0.1);
}

.filter-content {
    padding: 24px;
    display: none;
}

.filter-content.show {
    display: block;
    animation: slideDown 0.3s ease;
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.filter-form {
    display: grid;
    gap: 20px;
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.filter-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--gray-700);
}

.input-wrapper {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray-400);
    z-index: 2;
}

.form-control {
    width: 100%;
    padding: 12px 16px 12px 40px;
    border: 2px solid var(--gray-200);
    border-radius: 12px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-control:focus {
    outline: none;
    border-color: var(--primary);
}

select.form-control {
    appearance: none;
    background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' viewBox='0 0 12 12'%3E%3Cpath fill='%236b7280' d='M10.293 3.293L6 7.586 1.707 3.293A1 1 0 00.293 4.707l5 5a1 1 0 001.414 0l5-5a1 1 0 10-1.414-1.414z'/%3E%3C/svg%3E");
    background-repeat: no-repeat;
    background-position: right 12px center;
    padding-right: 40px;
}

.filter-actions {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.filter-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: 12px;
    font-weight: 600;
    font-size: 14px;
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-btn.search {
    background: var(--gradient);
    color: white;
}

.filter-btn.search:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 53, 0.3);
}

.filter-btn.clear {
    background: var(--gray-100);
    color: var(--gray-600);
}

.filter-btn.clear:hover {
    background: var(--gray-200);
}

/* Records Section */
.records-section {
    background: white;
    border-radius: 24px;
    padding: 32px;
    margin-bottom: 32px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
}

.section-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 24px;
    gap: 16px;
}

.section-title {
    font-size: 20px;
    font-weight: 700;
    color: var(--gray-900);
    margin: 0;
    display: flex;
    align-items: center;
    gap: 8px;
}

.section-title i {
    color: var(--primary);
}

.records-count {
    font-size: 14px;
    color: var(--gray-500);
    font-weight: 500;
}

.view-toggle {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 8px 16px;
    background: var(--gray-100);
    border: none;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 500;
    color: var(--gray-700);
    cursor: pointer;
    transition: all 0.3s ease;
}

.view-toggle:hover {
    background: var(--gray-200);
}

.records-list {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 20px;
}

.record-card {
    background: white;
    border: 2px solid var(--gray-100);
    border-radius: 16px;
    padding: 20px;
    transition: all 0.3s ease;
    position: relative;
}

.record-card:hover {
    border-color: var(--primary);
    box-shadow: 0 8px 32px rgba(255, 107, 53, 0.1);
    transform: translateY(-2px);
}

.record-card.income {
    border-left-color: #10B981;
    border-left-width: 4px;
}

.record-card.expense {
    border-left-color: #EF4444;
    border-left-width: 4px;
}

.record-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 16px;
}

.record-icon {
    width: 48px;
    height: 48px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    color: white;
}

.record-icon.income {
    background: linear-gradient(135deg, #10B981, #059669);
}

.record-icon.expense {
    background: linear-gradient(135deg, #EF4444, #DC2626);
}

.record-info {
    flex: 1;
    margin-left: 16px;
}

.record-category {
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: var(--gray-900);
    margin-bottom: 4px;
    font-size: 16px;
}

.record-amount {
    font-size: 18px;
    font-weight: 700;
}

.record-amount.income {
    color: #10B981;
}

.record-amount.expense {
    color: #EF4444;
}

.record-menu {
    position: relative;
}

.menu-btn {
    width: 32px;
    height: 32px;
    border: none;
    background: var(--gray-100);
    border-radius: 10px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--gray-500);
    transition: all 0.3s ease;
}

.menu-btn:hover {
    background: var(--gray-200);
}

.menu-dropdown {
    position: absolute;
    top: 100%;
    right: 0;
    background: white;
    border: 1px solid var(--gray-200);
    border-radius: 12px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    padding: 8px;
    min-width: 160px;
    z-index: 10;
    display: none;
}

.menu-dropdown.active {
    display: block;
}

.menu-item {
    width: 100%;
    padding: 8px 12px;
    border: none;
    background: none;
    text-align: left;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    color: var(--gray-700);
    display: flex;
    align-items: center;
    gap: 8px;
    transition: all 0.3s ease;
}

.menu-item:hover {
    background: var(--gray-50);
}

.menu-item.danger {
    color: #EF4444;
}

.menu-item.danger:hover {
    background: #FEF2F2;
}

.record-body {
    margin-top: 16px;
}

.record-note {
    font-size: 14px;
    color: var(--gray-600);
    margin-bottom: 12px;
    line-height: 1.5;
    display: flex;
    align-items: center;
    gap: 8px;
    flex-wrap: wrap;
}

.source-tag {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 8px;
    border-radius: 8px;
    font-size: 12px;
    font-weight: 600;
}

.source-tag.delivery {
    background: #E0F2FE;
    color: #0284C7;
}

.record-meta {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 12px;
    flex-wrap: wrap;
}

.record-date, .record-updated {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 12px;
    color: var(--gray-400);
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: var(--gray-500);
}

.empty-icon {
    width: 80px;
    height: 80px;
    background: var(--gradient-soft);
    border-radius: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 32px;
    color: var(--primary);
    margin: 0 auto 20px;
}

.empty-state h3 {
    font-size: 20px;
    font-weight: 600;
    color: var(--gray-700);
    margin-bottom: 8px;
}

.empty-state p {
    font-size: 14px;
    margin-bottom: 24px;
}

.empty-actions {
    display: flex;
    justify-content: center;
    gap: 12px;
    flex-wrap: wrap;
}

/* Pagination */
.pagination-container {
    margin-top: 32px;
    padding-top: 24px;
    border-top: 1px solid var(--gray-200);
}

/* Responsive */
@media (max-width: 768px) {
    .records-container {
        padding: 16px;
    }

    .records-header {
        padding: 24px;
    }

    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 20px;
    }

    .header-actions {
        width: 100%;
        justify-content: stretch;
    }

    .action-btn {
        flex: 1;
        justify-content: center;
    }

    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 16px;
    }

    .records-section {
        padding: 24px;
    }

    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 16px;
    }

    .records-list {
        grid-template-columns: 1fr;
        gap: 16px;
    }

    .filter-row {
        grid-template-columns: 1fr;
        gap: 16px;
    }

    .filter-actions {
        justify-content: stretch;
    }

    .filter-btn {
        flex: 1;
        justify-content: center;
    }

    .empty-actions {
        flex-direction: column;
        align-items: center;
    }

    .action-btn {
        width: 100%;
        max-width: 200px;
    }
}

@media (max-width: 480px) {
    .header-left {
        flex-direction: column;
        align-items: flex-start;
        gap: 12px;
    }

    .header-icon {
        width: 48px;
        height: 48px;
        font-size: 20px;
    }

    .records-title {
        font-size: 24px;
    }

    .stats-grid {
        grid-template-columns: 1fr;
    }

    .record-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 12px;
    }

    .record-info {
        margin-left: 0;
        width: 100%;
    }

    .record-meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 8px;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
// Toggle Filters
function toggleFilters() {
    const content = document.getElementById('filterContent');
    const toggle = document.querySelector('.filter-toggle i');
    
    if (content.style.display === 'none' || !content.style.display) {
        content.style.display = 'block';
        content.classList.add('show');
        toggle.style.transform = 'rotate(180deg)';
    } else {
        content.style.display = 'none';
        content.classList.remove('show');
        toggle.style.transform = 'rotate(0deg)';
    }
}

// Toggle Record Menu
function toggleRecordMenu(recordId) {
    const menu = document.getElementById(`menu-${recordId}`);
    document.querySelectorAll('.menu-dropdown').forEach(m => {
        if (m.id !== `menu-${recordId}`) {
            m.classList.remove('active');
        }
    });
    menu.classList.toggle('active');
}

// Close menus when clicking outside
document.addEventListener('click', function(e) {
    if (!e.target.closest('.record-menu')) {
        document.querySelectorAll('.menu-dropdown').forEach(m => {
            m.classList.remove('active');
        });
    }
});

// Toggle View
function toggleView() {
    const list = document.getElementById('recordsList');
    const toggle = document.getElementById('viewToggle');
    
    if (list.style.display === 'block' || !list.style.display) {
        list.style.display = 'grid';
        toggle.innerHTML = '<i class="fas fa-list"></i><span>มุมมองรายการ</span>';
    } else {
        list.style.display = 'block';
        toggle.innerHTML = '<i class="fas fa-th-large"></i><span>มุมมองการ์ด</span>';
    }
}

// Edit Record
function editRecord(id) {
    window.location.href = `{{ route('finance.records.edit', ':id') }}`.replace(':id', id);
}

// Delete Record
function deleteRecord(id) {
    Swal.fire({
        title: 'ยืนยันการลบ',
        text: 'คุณต้องการลบรายการนี้หรือไม่?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#EF4444',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'ลบ',
        cancelButtonText: 'ยกเลิก'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`{{ route('finance.records.destroy', ':id') }}`.replace(':id', id), {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire({
                        icon: 'success',
                        title: 'สำเร็จ',
                        text: 'ลบรายการเรียบร้อยแล้ว',
                        timer: 1500,
                        showConfirmButton: false
                    }).then(() => location.reload());
                } else {
                    Swal.fire('เกิดข้อผิดพลาด', data.message || 'ไม่สามารถลบรายการได้', 'error');
                }
            })
            .catch(() => Swal.fire('เกิดข้อผิดพลาด', 'ไม่สามารถเชื่อมต่อเซิร์ฟเวอร์ได้', 'error'));
        }
    });
    
    document.querySelectorAll('.menu-dropdown').forEach(m => {
        m.classList.remove('active');
    });
}

// Duplicate Record
function duplicateRecord(id) {
    Swal.fire({
        title: 'คัดลอกรายการ',
        text: 'คุณต้องการคัดลอกรายการนี้หรือไม่?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#3B82F6',
        cancelButtonColor: '#6B7280',
        confirmButtonText: 'คัดลอก',
        cancelButtonText: 'ยกเลิก'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch(`{{ route('finance.records.duplicate', ':id') }}`.replace(':id', id), {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    Swal.fire({
                        icon: 'success',
                        title: 'สำเร็จ',
                        text: 'คัดลอกรายการเรียบร้อยแล้ว',
                        timer: 1500,
                        showConfirmButton: false
                    }).then(() => {
                        if (data.redirect) {
                            window.location.href = data.redirect;
                        } else {
                            location.reload();
                        }
                    });
                } else {
                    Swal.fire('เกิดข้อผิดพลาด', data.message || 'ไม่สามารถคัดลอกรายการได้', 'error');
                }
            })
            .catch(() => Swal.fire('เกิดข้อผิดพลาด', 'ไม่สามารถเชื่อมต่อเซิร์ฟเวอร์ได้', 'error'));
        }
    });
    
    document.querySelectorAll('.menu-dropdown').forEach(m => {
        m.classList.remove('active');
    });
}

// Export Records
function exportRecords() {
    Swal.fire({
        title: 'เลือกรูปแบบการส่งออก',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'CSV (Excel)',
        cancelButtonText: 'PDF',
        confirmButtonColor: '#10B981',
        cancelButtonColor: '#3B82F6',
    }).then((result) => {
        if (result.isConfirmed) {
            window.location.href = '{{ route('finance.reports.export') }}?type=monthly&format=csv&period={{ now()->format('Y-m') }}';
        } else if (result.dismiss === Swal.DismissReason.cancel) {
            window.location.href = '{{ route('finance.reports.export') }}?type=monthly&format=pdf&period={{ now()->format('Y-m') }}';
        }
    });
}

// Session Messages
@if (session('success'))
    Swal.fire({
        icon: 'success',
        title: 'สำเร็จ!',
        text: '{{ session('success') }}',
        confirmButtonColor: '#FF6B35'
    });
@endif

@if (session('error'))
    Swal.fire({
        icon: 'error',
        title: 'เกิดข้อผิดพลาด!',
        text: '{{ session('error') }}',
        confirmButtonColor: '#FF6B35'
    });
@endif

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    console.log('Records page initialized');
});
</script>
@endpush
@endsection