@extends('layouts.app')

@section('page-title', 'เพิ่มรายการ')
@section('mobile-title', 'บันทึกรายการใหม่')

@section('content')
<div class="create-record-container">
    <!-- Header Section -->
    <div class="page-header-modern">
        <div class="header-content">
            <div class="header-left">
                <div class="page-icon">
                    <i class="fas fa-plus-circle"></i>
                </div>
                <div class="page-info">
                    <h1 class="page-title">บันทึกรายการใหม่</h1>
                    <p class="page-subtitle">เพิ่มรายรับหรือรายจ่ายของคุณ</p>
                </div>
            </div>
            <div class="header-actions">
                <a href="{{ route('finance.records.index') }}" class="action-btn secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span>กลับ</span>
                </a>
            </div>
        </div>
    </div>

    <form method="POST" action="{{ route('finance.records.store') }}" class="record-form" id="recordForm">
        @csrf

        <!-- Type Selection -->
        <div class="type-selection">
            <h3 class="section-title">เลือกประเภทรายการ</h3>
            <div class="type-cards">
                <label class="type-card {{ old('type', $type) === 'income' ? 'selected' : '' }}">
                    <input type="radio" name="type" value="income" 
                           {{ old('type', $type) === 'income' ? 'checked' : '' }}
                           onchange="handleTypeChange(this.value)">
                    <div class="card-content income">
                        <div class="card-icon">
                            <i class="fas fa-arrow-trend-up"></i>
                        </div>
                        <div class="card-info">
                            <h4>รายรับ</h4>
                            <p>บันทึกเงินที่ได้รับ</p>
                        </div>
                    </div>
                </label>
                
                <label class="type-card {{ old('type', $type) === 'expense' ? 'selected' : '' }}">
                    <input type="radio" name="type" value="expense" 
                           {{ old('type', $type) === 'expense' ? 'checked' : '' }}
                           onchange="handleTypeChange(this.value)">
                    <div class="card-content expense">
                        <div class="card-icon">
                            <i class="fas fa-arrow-trend-down"></i>
                        </div>
                        <div class="card-info">
                            <h4>รายจ่าย</h4>
                            <p>บันทึกค่าใช้จ่าย</p>
                        </div>
                    </div>
                </label>
            </div>
        </div>

        <!-- Form Fields -->
        <div class="form-section">
            <div class="form-grid">
                <!-- Date -->
                <div class="form-field">
                    <label class="field-label">
                        <i class="fas fa-calendar"></i>
                        วันที่
                        <span class="required">*</span>
                    </label>
                    <input type="date" 
                           name="record_date" 
                           class="field-input @error('record_date') error @enderror"
                           value="{{ old('record_date', date('Y-m-d')) }}"
                           max="{{ date('Y-m-d') }}"
                           required>
                    @error('record_date')
                        <span class="field-error">{{ $message }}</span>
                    @enderror
                </div>

                <!-- Amount -->
                <div class="form-field">
                    <label class="field-label">
                        <i class="fas fa-coins"></i>
                        จำนวนเงิน
                        <span class="required">*</span>
                    </label>
                    <div class="amount-input-wrapper">
                        <span class="currency-symbol">฿</span>
                        <input type="number" 
                               name="amount" 
                               class="field-input amount-input @error('amount') error @enderror"
                               value="{{ old('amount') }}"
                               step="0.01"
                               min="0.01"
                               max="9999999.99"
                               placeholder="0.00"
                               required>
                    </div>
                    @error('amount')
                        <span class="field-error">{{ $message }}</span>
                    @enderror
                </div>

                <!-- Category -->
                <div class="form-field full-width">
                    <label class="field-label">
                        <i class="fas fa-tag"></i>
                        หมวดหมู่
                        <span class="required">*</span>
                    </label>
                    <select name="category" 
                            class="field-input category-select @error('category') error @enderror"
                            id="categorySelect"
                            required>
                        <option value="">เลือกหมวดหมู่</option>
                        @foreach($categories as $value => $name)
                            <option value="{{ $value }}" 
                                    {{ old('category') == $value ? 'selected' : '' }}>
                                {{ $name }}
                            </option>
                        @endforeach
                    </select>
                    @error('category')
                        <span class="field-error">{{ $message }}</span>
                    @enderror
                </div>

                <!-- Note -->
                <div class="form-field full-width">
                    <label class="field-label">
                        <i class="fas fa-sticky-note"></i>
                        หมายเหตุ
                    </label>
                    <textarea name="note" 
                              class="field-input note-input @error('note') error @enderror"
                              rows="3"
                              maxlength="500"
                              placeholder="รายละเอียดเพิ่มเติม (ถ้ามี)">{{ old('note') }}</textarea>
                    @error('note')
                        <span class="field-error">{{ $message }}</span>
                    @enderror
                    <div class="char-counter">
                        <span id="charCount">0</span>/500 ตัวอักษร
                    </div>
                </div>
            </div>

            <!-- Recurring Options -->
            <div class="recurring-section">
                <label class="recurring-toggle">
                    <input type="checkbox" 
                           name="is_recurring" 
                           value="1"
                           id="recurringCheck"
                           {{ old('is_recurring') ? 'checked' : '' }}>
                    <span class="toggle-slider"></span>
                    <div class="toggle-content">
                        <h4>รายการซ้ำ</h4>
                        <p>สร้างรายการล่วงหน้าอัตโนมัติ</p>
                    </div>
                </label>
                
                <div id="recurringOptions" class="recurring-options" style="display: {{ old('is_recurring') ? 'block' : 'none' }};">
                    <div class="form-field">
                        <label class="field-label">
                            <i class="fas fa-clock"></i>
                            ความถี่
                        </label>
                        <select name="recurring_type" class="field-input">
                            <option value="daily" {{ old('recurring_type') == 'daily' ? 'selected' : '' }}>
                                ทุกวัน
                            </option>
                            <option value="weekly" {{ old('recurring_type') == 'weekly' ? 'selected' : '' }}>
                                ทุกสัปดาห์
                            </option>
                            <option value="monthly" {{ old('recurring_type') == 'monthly' ? 'selected' : '' }}>
                                ทุกเดือน
                            </option>
                            <option value="yearly" {{ old('recurring_type') == 'yearly' ? 'selected' : '' }}>
                                ทุกปี
                            </option>
                        </select>
                        <div class="field-hint">
                            <i class="fas fa-info-circle"></i>
                            ระบบจะสร้างรายการล่วงหน้า 3 เดือน
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Form Actions -->
        <div class="form-actions">
            <button type="submit" class="btn-save">
                <i class="fas fa-save"></i>
                <span>บันทึกรายการ</span>
            </button>
            <button type="button" class="btn-draft" onclick="saveDraft()">
                <i class="fas fa-file-alt"></i>
                <span>บันทึกร่าง</span>
            </button>
            <a href="{{ route('finance.records.index') }}" class="btn-cancel">
                <i class="fas fa-times"></i>
                <span>ยกเลิก</span>
            </a>
        </div>
    </form>

    <!-- Quick Stats -->
    <div class="quick-stats">
        <h3 class="stats-title">
            <i class="fas fa-chart-line"></i>
            สถิติเดือนนี้
        </h3>
        <div class="stats-grid">
            <div class="stat-card income">
                <div class="stat-icon">
                    <i class="fas fa-arrow-up"></i>
                </div>
                <div class="stat-content">
                    <span class="stat-value">฿0</span>
                    <span class="stat-label">รายรับ</span>
                </div>
            </div>
            
            <div class="stat-card expense">
                <div class="stat-icon">
                    <i class="fas fa-arrow-down"></i>
                </div>
                <div class="stat-content">
                    <span class="stat-value">฿0</span>
                    <span class="stat-label">รายจ่าย</span>
                </div>
            </div>
            
            <div class="stat-card balance">
                <div class="stat-icon">
                    <i class="fas fa-balance-scale"></i>
                </div>
                <div class="stat-content">
                    <span class="stat-value">฿0</span>
                    <span class="stat-label">คงเหลือ</span>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
/* Container */
.create-record-container {
    max-width: 100%;
    margin: 0;
    padding: 0;
}

/* Page Header */
.page-header-modern {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8A65 100%);
    border-radius: 24px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 8px 32px rgba(255, 107, 53, 0.15);
    position: relative;
    overflow: hidden;
}

.page-header-modern::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -20%;
    width: 300px;
    height: 300px;
    background: radial-gradient(circle, rgba(255, 255, 255, 0.1) 0%, transparent 70%);
    border-radius: 50%;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    z-index: 2;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.page-icon {
    width: 60px;
    height: 60px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
    backdrop-filter: blur(10px);
}

.page-info .page-title {
    font-size: 1.75rem;
    font-weight: 700;
    color: white;
    margin: 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.page-info .page-subtitle {
    font-size: 1rem;
    color: rgba(255, 255, 255, 0.9);
    margin: 0.25rem 0 0 0;
    font-weight: 400;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.25rem;
    border-radius: 12px;
    font-weight: 600;
    font-size: 0.9rem;
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-btn.secondary {
    background: rgba(255, 255, 255, 0.2);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.3);
    backdrop-filter: blur(10px);
}

.action-btn.secondary:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-1px);
}

/* Form */
.record-form {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
}

/* Type Selection */
.type-selection {
    margin-bottom: 2rem;
}

.section-title {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1F2937;
    margin-bottom: 1rem;
}

.type-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.type-card {
    cursor: pointer;
    border-radius: 16px;
    border: 2px solid #E5E7EB;
    transition: all 0.2s ease;
    overflow: hidden;
}

.type-card input {
    display: none;
}

.type-card.selected {
    border-color: #FF6B35;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.2);
}

.card-content {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1.5rem;
}

.card-content.income {
    background: linear-gradient(135deg, #D1FAE5 0%, #A7F3D0 100%);
}

.card-content.expense {
    background: linear-gradient(135deg, #FEE2E2 0%, #FECACA 100%);
}

.card-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    background: white;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.card-content.income .card-icon {
    color: #059669;
}

.card-content.expense .card-icon {
    color: #DC2626;
}

.card-info h4 {
    font-size: 1.1rem;
    font-weight: 600;
    color: #1F2937;
    margin: 0 0 0.25rem 0;
}

.card-info p {
    font-size: 0.9rem;
    color: #6B7280;
    margin: 0;
}

/* Form Section */
.form-section {
    margin-bottom: 2rem;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.form-field {
    display: flex;
    flex-direction: column;
}

.form-field.full-width {
    grid-column: span 2;
}

.field-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    font-size: 0.9rem;
}

.required {
    color: #DC2626;
}

.field-input {
    padding: 0.75rem 1rem;
    border: 2px solid #E5E7EB;
    border-radius: 12px;
    font-size: 0.9rem;
    transition: all 0.2s ease;
    background: white;
}

.field-input:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.field-input.error {
    border-color: #DC2626;
}

.field-error {
    color: #DC2626;
    font-size: 0.8rem;
    margin-top: 0.25rem;
    font-weight: 500;
}

/* Amount Input */
.amount-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
    border: 2px solid #E5E7EB;
    border-radius: 12px;
    overflow: hidden;
    transition: all 0.2s ease;
}

.amount-input-wrapper:focus-within {
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.currency-symbol {
    background: #F9FAFB;
    padding: 0.75rem 1rem;
    font-weight: 600;
    color: #6B7280;
    border-right: 1px solid #E5E7EB;
}

.amount-input {
    border: none !important;
    box-shadow: none !important;
    flex: 1;
}

/* Note Input */
.note-input {
    resize: vertical;
    min-height: 80px;
}

.char-counter {
    font-size: 0.8rem;
    color: #6B7280;
    margin-top: 0.25rem;
    text-align: right;
}

/* Recurring Section */
.recurring-section {
    background: #F9FAFB;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 1rem;
}

.recurring-toggle {
    display: flex;
    align-items: center;
    gap: 1rem;
    cursor: pointer;
}

.recurring-toggle input {
    display: none;
}

.toggle-slider {
    width: 48px;
    height: 24px;
    background: #E5E7EB;
    border-radius: 12px;
    position: relative;
    transition: all 0.2s ease;
}

.toggle-slider::before {
    content: '';
    position: absolute;
    top: 2px;
    left: 2px;
    width: 20px;
    height: 20px;
    background: white;
    border-radius: 50%;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.recurring-toggle input:checked + .toggle-slider {
    background: #FF6B35;
}

.recurring-toggle input:checked + .toggle-slider::before {
    transform: translateX(24px);
}

.toggle-content h4 {
    font-size: 1rem;
    font-weight: 600;
    color: #1F2937;
    margin: 0 0 0.25rem 0;
}

.toggle-content p {
    font-size: 0.85rem;
    color: #6B7280;
    margin: 0;
}

.recurring-options {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid #E5E7EB;
}

.field-hint {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    font-size: 0.8rem;
    color: #6B7280;
    margin-top: 0.25rem;
}

/* Form Actions */
.form-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.btn-save, .btn-draft, .btn-cancel {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 600;
    font-size: 0.9rem;
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-save {
    background: #FF6B35;
    color: white;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.btn-save:hover {
    background: #E85D2F;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 107, 53, 0.4);
}

.btn-draft {
    background: #F3F4F6;
    color: #6B7280;
    border: 1px solid #E5E7EB;
}

.btn-draft:hover {
    background: #E5E7EB;
    color: #374151;
}

.btn-cancel {
    background: #FEF2F2;
    color: #DC2626;
    border: 1px solid #FECACA;
}

.btn-cancel:hover {
    background: #FEE2E2;
    color: #B91C1C;
}

/* Quick Stats */
.quick-stats {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
}

.stats-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.25rem;
    font-weight: 700;
    color: #1F2937;
    margin: 0 0 1.5rem 0;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 1rem;
}

.stat-card {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border-radius: 12px;
    border: 1px solid #E5E7EB;
    transition: all 0.2s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1rem;
}

.stat-card.income .stat-icon {
    background: linear-gradient(135deg, #D1FAE5 0%, #A7F3D0 100%);
    color: #059669;
}

.stat-card.expense .stat-icon {
    background: linear-gradient(135deg, #FEE2E2 0%, #FECACA 100%);
    color: #DC2626;
}

.stat-card.balance .stat-icon {
    background: linear-gradient(135deg, #DBEAFE 0%, #BFDBFE 100%);
    color: #2563EB;
}

.stat-value {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1F2937;
    display: block;
}

.stat-label {
    font-size: 0.8rem;
    color: #6B7280;
    display: block;
}

/* Responsive Design */
@media (max-width: 768px) {
    .page-header-modern {
        padding: 1.5rem;
    }
    
    .header-content {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .header-left {
        flex-direction: column;
        gap: 1rem;
    }
    
    .record-form {
        padding: 1.5rem;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .form-field.full-width {
        grid-column: span 1;
    }
    
    .type-cards {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .btn-save, .btn-draft, .btn-cancel {
        justify-content: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 480px) {
    .page-header-modern {
        padding: 1rem;
        border-radius: 16px;
    }
    
    .page-icon {
        width: 48px;
        height: 48px;
        font-size: 1.25rem;
    }
    
    .page-info .page-title {
        font-size: 1.5rem;
    }
    
    .record-form, .quick-stats {
        padding: 1rem;
    }
    
    .card-content {
        padding: 1rem;
    }
    
    .recurring-section {
        padding: 1rem;
    }
}
</style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('recordForm');
    const typeInputs = document.querySelectorAll('input[name="type"]');
    const categorySelect = document.getElementById('categorySelect');
    const recurringCheck = document.getElementById('recurringCheck');
    const recurringOptions = document.getElementById('recurringOptions');
    const noteTextarea = document.querySelector('.note-input');
    const charCount = document.getElementById('charCount');
    const amountInput = document.querySelector('.amount-input');

    // Handle type change
    window.handleTypeChange = function(type) {
        // Update visual selection
        document.querySelectorAll('.type-card').forEach(card => {
            card.classList.remove('selected');
        });
        document.querySelector(`input[value="${type}"]`).closest('.type-card').classList.add('selected');
        
        // Load categories
        loadCategories(type);
    }

    // Load categories based on type
    function loadCategories(type) {
        if (!type) return;
        
        fetch(`{{ route('finance.records.categories') }}?type=${type}`)
            .then(response => response.json())
            .then(categories => {
                categorySelect.innerHTML = '<option value="">เลือกหมวดหมู่</option>';
                categories.forEach(cat => {
                    const option = document.createElement('option');
                    option.value = cat.slug;
                    option.textContent = cat.name;
                    categorySelect.appendChild(option);
                });
            })
            .catch(error => {
                console.error('Error loading categories:', error);
                if (window.Swal) {
                    Swal.fire('เกิดข้อผิดพลาด', 'ไม่สามารถโหลดหมวดหมู่ได้', 'error');
                }
            });
    }

    // Initialize with selected type
    const selectedType = document.querySelector('input[name="type"]:checked');
    if (selectedType) {
        handleTypeChange(selectedType.value);
    }

    // Recurring checkbox
    recurringCheck.addEventListener('change', function() {
        recurringOptions.style.display = this.checked ? 'block' : 'none';
    });

    // Character counter
    if (noteTextarea && charCount) {
        noteTextarea.addEventListener('input', function() {
            const length = this.value.length;
            charCount.textContent = length;
            
            if (length > 400) {
                charCount.style.color = '#DC2626';
            } else if (length > 300) {
                charCount.style.color = '#F59E0B';
            } else {
                charCount.style.color = '#6B7280';
            }
        });
        
        // Initialize counter
        charCount.textContent = noteTextarea.value.length;
    }

    // Amount input formatting
    if (amountInput) {
        amountInput.addEventListener('input', function() {
            // Remove non-numeric except decimal point
            this.value = this.value.replace(/[^0-9.]/g, '');
            
            // Ensure only one decimal point
            const parts = this.value.split('.');
            if (parts.length > 2) {
                this.value = parts[0] + '.' + parts.slice(1).join('');
            }
            
            // Limit to 2 decimal places
            if (parts[1] && parts[1].length > 2) {
                this.value = parts[0] + '.' + parts[1].substring(0, 2);
            }
        });

        amountInput.addEventListener('blur', function() {
            if (this.value && !isNaN(this.value)) {
                const value = parseFloat(this.value);
                this.value = value.toFixed(2);
            }
        });
    }

    // Form validation
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const type = document.querySelector('input[name="type"]:checked');
        const category = categorySelect.value;
        const amount = amountInput.value;
        const date = document.querySelector('input[name="record_date"]').value;
        
        // Validate
        if (!type) {
            if (window.Swal) {
                Swal.fire('กรุณาเลือกประเภท', 'เลือกว่าเป็นรายรับหรือรายจ่าย', 'warning');
            } else {
                alert('กรุณาเลือกประเภท');
            }
            return;
        }
        
        if (!category) {
            if (window.Swal) {
                Swal.fire('กรุณาเลือกหมวดหมู่', 'เลือกหมวดหมู่ของรายการ', 'warning');
            } else {
                alert('กรุณาเลือกหมวดหมู่');
            }
            categorySelect.focus();
            return;
        }
        
        if (!amount || parseFloat(amount) <= 0) {
            if (window.Swal) {
                Swal.fire('กรุณาระบุจำนวนเงิน', 'จำนวนเงินต้องมากกว่า 0', 'warning');
            } else {
                alert('กรุณาระบุจำนวนเงิน');
            }
            amountInput.focus();
            return;
        }
        
        if (!date) {
            if (window.Swal) {
                Swal.fire('กรุณาเลือกวันที่', 'เลือกวันที่ของรายการ', 'warning');
            } else {
                alert('กรุณาเลือกวันที่');
            }
            return;
        }
        
        // Show loading
        if (window.Swal) {
            Swal.fire({
                title: 'กำลังบันทึก...',
                text: 'กรุณารอสักครู่',
                allowOutsideClick: false,
                showConfirmButton: false,
                willOpen: () => {
                    Swal.showLoading();
                }
            });
        }
        
        // Submit form
        this.submit();
    });

    // Save draft function
    window.saveDraft = function() {
        const formData = {
            type: document.querySelector('input[name="type"]:checked')?.value,
            date: document.querySelector('input[name="record_date"]').value,
            category: categorySelect.value,
            amount: amountInput.value,
            note: noteTextarea.value,
            is_recurring: recurringCheck.checked,
            recurring_type: document.querySelector('select[name="recurring_type"]').value
        };
        
        localStorage.setItem('finance_record_draft', JSON.stringify(formData));
        
        if (window.Swal) {
            Swal.fire({
                icon: 'success',
                title: 'บันทึกร่างเรียบร้อย',
                text: 'ข้อมูลถูกบันทึกไว้ชั่วคราว',
                timer: 1500,
                showConfirmButton: false
            });
        } else {
            alert('บันทึกร่างเรียบร้อย');
        }
    }

    // Load draft on page load
    const draft = localStorage.getItem('finance_record_draft');
    if (draft && !{{ old('type') ? 'true' : 'false' }}) {
        if (window.Swal) {
            Swal.fire({
                title: 'พบข้อมูลที่บันทึกไว้',
                text: 'ต้องการโหลดข้อมูลที่บันทึกไว้หรือไม่?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'โหลด',
                cancelButtonText: 'ไม่'
            }).then((result) => {
                if (result.isConfirmed) {
                    loadDraftData(JSON.parse(draft));
                }
            });
        } else {
            if (confirm('พบข้อมูลที่บันทึกไว้ ต้องการโหลดหรือไม่?')) {
                loadDraftData(JSON.parse(draft));
            }
        }
    }

    function loadDraftData(data) {
        if (data.type) {
            document.querySelector(`input[name="type"][value="${data.type}"]`).checked = true;
            handleTypeChange(data.type);
        }
        if (data.date) {
            document.querySelector('input[name="record_date"]').value = data.date;
        }
        if (data.category) {
            setTimeout(() => {
                categorySelect.value = data.category;
            }, 500);
        }
        if (data.amount) {
            amountInput.value = data.amount;
        }
        if (data.note) {
            noteTextarea.value = data.note;
            charCount.textContent = data.note.length;
        }
        if (data.is_recurring) {
            recurringCheck.checked = true;
            recurringOptions.style.display = 'block';
        }
        if (data.recurring_type) {
            document.querySelector('select[name="recurring_type"]').value = data.recurring_type;
        }
    }

    // Clear draft after successful submission
    @if(session('success'))
        localStorage.removeItem('finance_record_draft');
    @endif

    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        // Ctrl/Cmd + S: Save
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
            e.preventDefault();
            form.querySelector('button[type="submit"]').click();
        }
        
        // Ctrl/Cmd + D: Save draft
        if ((e.ctrlKey || e.metaKey) && e.key === 'd') {
            e.preventDefault();
            saveDraft();
        }
    });
});
</script>
@endpush