<div class="notification-bell" x-data="notificationBell()">
    <!-- Bell Button -->
    <button @click="toggleDropdown()" class="bell-button" title="การแจ้งเตือน">
        <i class="fas fa-bell"></i>
        <div x-show="unreadCount > 0" 
             x-text="unreadCount > 99 ? '99+' : unreadCount" 
             class="notification-badge"></div>
    </button>

    <!-- Dropdown -->
    <div x-show="isOpen" 
         @click.away="isOpen = false"
         x-transition:enter="transform ease-out duration-400"
         x-transition:enter-start="opacity-0 scale-95 translate-y-[-10px]"
         x-transition:enter-end="opacity-100 scale-100 translate-y-8px"
         x-transition:leave="transition ease-in duration-200"
         x-transition:leave-start="opacity-100 scale-100"
         x-transition:leave-end="opacity-0 scale-95"
         class="notification-dropdown" :class="{ 'active': isOpen }">
            <!-- Header -->
            <div class="notification-header">
                <h3>การแจ้งเตือน</h3>
                <p class="notification-subtitle" x-text="unreadCount > 0 ? `คุณมี ${unreadCount} การแจ้งเตือนใหม่` : 'ไม่มีการแจ้งเตือนใหม่'"></p>
                <div class="header-actions">
                    <button @click="markAllAsRead()" class="action-btn" x-show="unreadCount > 0" title="ทำเครื่องหมายอ่านทั้งหมด">
                        <i class="fas fa-check"></i>
                    </button>
                    <button class="action-btn" title="ตั้งค่า">
                        <i class="fas fa-cog"></i>
                    </button>
                </div>
            </div>

            <!-- Notifications List -->
            <div class="notification-scroll">
                <template x-for="notification in notifications" :key="notification.id">
                    <div class="notification-item" :class="{ 'unread': !notification.read_at }" @click="markAsRead(notification)">
                        <div class="notification-icon" :class="notification.type">
                            <i class="fas" :class="{
                                'fa-check-circle': notification.type === 'success',
                                'fa-exclamation-triangle': notification.type === 'warning',
                                'fa-times-circle': notification.type === 'error',
                                'fa-info-circle': notification.type === 'info',
                                'fa-truck': notification.type === 'delivery',
                                'fa-coins': notification.type === 'commission'
                            }"></i>
                        </div>
                        <div class="notification-content">
                            <div class="notification-title" x-text="notification.title"></div>
                            <div class="notification-message" x-text="notification.message"></div>
                            <div class="notification-meta">
                                <span class="notification-time" x-text="formatTime(notification.created_at)"></span>
                                <span class="notification-type-badge" :class="'type-' + notification.type" x-text="{
                                    'success': 'สำเร็จ',
                                    'warning': 'เตือน', 
                                    'error': 'ข้อผิดพลาด',
                                    'info': 'ข้อมูล',
                                    'delivery': 'การจัดส่ง',
                                    'commission': 'คอมมิชชั่น'
                                }[notification.type] || 'ทั่วไป'"></span>
                                <div x-show="!notification.read_at" class="unread-dot"></div>
                            </div>
                        </div>
                    </div>
                </template>

                <!-- Empty State -->
                <div x-show="notifications.length === 0" class="notification-empty">
                    <div class="notification-empty-icon">
                        <i class="fas fa-bell-slash"></i>
                    </div>
                    <h4>ไม่มีการแจ้งเตือน</h4>
                    <p>คุณไม่มีการแจ้งเตือนใหม่ในขณะนี้</p>
                </div>
            </div>
    </div>
</div>

<script>
function notificationBell() {
    return {
        isOpen: false,
        notifications: [],
        unreadCount: 0,

        init() {
            this.loadNotifications();
            this.loadUnreadCount();
            
            // Poll for new notifications every 30 seconds
            setInterval(() => {
                this.loadUnreadCount();
                if (this.isOpen) {
                    this.loadNotifications();
                }
            }, 30000);
        },

        toggleDropdown() {
            this.isOpen = !this.isOpen;
            if (this.isOpen) {
                this.loadNotifications();
            }
        },

        async loadNotifications() {
            try {
                const response = await fetch('/api/notifications');
                this.notifications = await response.json();
            } catch (error) {
                console.error('Error loading notifications:', error);
            }
        },

        async loadUnreadCount() {
            try {
                const response = await fetch('/api/notifications/unread-count');
                const data = await response.json();
                const oldCount = this.unreadCount;
                this.unreadCount = data.count;
                
                // เล่นเสียงถ้ามีการแจ้งเตือนใหม่
                if (data.count > oldCount && oldCount >= 0) {
                    this.playNotificationSound();
                }
            } catch (error) {
                console.error('Error loading unread count:', error);
            }
        },

        async markAsRead(notification) {
            if (notification.read_at) return;

            try {
                await fetch(`/api/notifications/${notification.id}/read`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                });
                
                notification.read_at = new Date().toISOString();
                this.unreadCount = Math.max(0, this.unreadCount - 1);
            } catch (error) {
                console.error('Error marking as read:', error);
            }
        },

        playNotificationSound() {
            try {
                // เพิ่ม bell animation
                const bellIcon = document.querySelector('.header-btn .fa-bell');
                if (bellIcon) {
                    bellIcon.parentElement.classList.add('notification-bell-active');
                    setTimeout(() => {
                        bellIcon.parentElement.classList.remove('notification-bell-active');
                    }, 1500);
                }
                
                // ใช้ Web Audio API ก่อน
                if (window.notificationSound) {
                    window.notificationSound.play('notification');
                    return;
                }
                
                // fallback ใช้ HTML5 Audio
                const audio = new Audio('/sounds/notification.mp3');
                audio.volume = 0.3;
                audio.play().catch(() => {});
            } catch (e) {}
        },

        async markAllAsRead() {
            try {
                await fetch('/api/notifications/mark-all-read', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                });
                
                this.notifications.forEach(n => n.read_at = new Date().toISOString());
                this.unreadCount = 0;
            } catch (error) {
                console.error('Error marking all as read:', error);
            }
        },

        async deleteNotification(notification) {
            try {
                await fetch(`/api/notifications/${notification.id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                });
                
                this.notifications = this.notifications.filter(n => n.id !== notification.id);
                if (!notification.read_at) {
                    this.unreadCount = Math.max(0, this.unreadCount - 1);
                }
            } catch (error) {
                console.error('Error deleting notification:', error);
            }
        },

        formatTime(dateString) {
            const date = new Date(dateString);
            const now = new Date();
            const diff = now - date;
            
            if (diff < 60000) return 'เมื่อสักครู่';
            if (diff < 3600000) return Math.floor(diff / 60000) + ' นาทีที่แล้ว';
            if (diff < 86400000) return Math.floor(diff / 3600000) + ' ชั่วโมงที่แล้ว';
            return Math.floor(diff / 86400000) + ' วันที่แล้ว';
        }
    }
}
</script>