@props(['class' => ''])

<div class="language-switcher {{ $class }}" x-data="languageSwitcher()">
    <button @click="toggleDropdown()" 
            class="flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
        <i class="fas fa-globe text-gray-500"></i>
        <span x-text="currentLanguage"></span>
        <i class="fas fa-chevron-down text-xs text-gray-400" :class="{ 'rotate-180': isOpen }"></i>
    </button>
    
    <div x-show="isOpen" 
         x-transition:enter="transition ease-out duration-100"
         x-transition:enter-start="transform opacity-0 scale-95"
         x-transition:enter-end="transform opacity-100 scale-100"
         x-transition:leave="transition ease-in duration-75"
         x-transition:leave-start="transform opacity-100 scale-100"
         x-transition:leave-end="transform opacity-0 scale-95"
         @click.away="isOpen = false"
         class="absolute right-0 z-50 mt-2 w-48 bg-white rounded-lg shadow-lg border border-gray-200 py-1">
        
        @foreach(config('app.available_locales') as $locale)
            <button @click="switchLanguage('{{ $locale }}')"
                    class="flex items-center w-full px-4 py-2 text-sm text-gray-700 hover:bg-gray-100 transition-colors"
                    :class="{ 'bg-primary-50 text-primary-700': currentLocale === '{{ $locale }}' }">
                <span class="mr-3">
                    @if($locale === 'th')
                        🇹🇭
                    @else
                        🇺🇸
                    @endif
                </span>
                {{ config('app.locale_names')[$locale] ?? $locale }}
                <i x-show="currentLocale === '{{ $locale }}'" class="fas fa-check ml-auto text-primary-600"></i>
            </button>
        @endforeach
    </div>
</div>

<script>
function languageSwitcher() {
    return {
        isOpen: false,
        currentLocale: '{{ app()->getLocale() }}',
        currentLanguage: '{{ config('app.locale_names')[app()->getLocale()] ?? app()->getLocale() }}',
        
        toggleDropdown() {
            this.isOpen = !this.isOpen;
        },
        
        async switchLanguage(locale) {
            this.isOpen = false;
            
            try {
                // Show loading state
                const originalText = this.currentLanguage;
                this.currentLanguage = '{{ __('common.loading') }}';
                
                // Make AJAX request
                const response = await fetch('{{ route('language.switch') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    },
                    body: JSON.stringify({ locale: locale })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Update localStorage
                    localStorage.setItem('locale', locale);
                    
                    // Show success message
                    if (window.Swal) {
                        Swal.fire({
                            toast: true,
                            position: 'top-end',
                            showConfirmButton: false,
                            timer: 2000,
                            timerProgressBar: true,
                            icon: 'success',
                            title: data.message
                        });
                    }
                    
                    // Reload page to apply new language
                    setTimeout(() => {
                        window.location.reload();
                    }, 500);
                } else {
                    throw new Error(data.message);
                }
            } catch (error) {
                // Restore original text
                this.currentLanguage = originalText;
                
                // Show error message
                if (window.Swal) {
                    Swal.fire({
                        toast: true,
                        position: 'top-end',
                        showConfirmButton: false,
                        timer: 3000,
                        icon: 'error',
                        title: error.message || '{{ __('common.error') }}'
                    });
                }
            }
        }
    }
}
</script>

<style>
.language-switcher {
    position: relative;
}

.rotate-180 {
    transform: rotate(180deg);
}

:root {
    --primary-50: #fff5f1;
    --primary-500: #ff6b35;
    --primary-600: #e85d2f;
    --primary-700: #d14d26;
}
</style>