@extends('layouts.app')

@section('title', 'รายงานการจัดส่ง - ระบบจัดการค่าคอมมิชชั่น')

@section('content')
<div class="delivery-reports-container">
    <!-- Header Card -->
    <div class="header-card">
        <div class="header-content">
            <div class="header-info">
                <div class="header-icon">
                    <i class="fas fa-truck"></i>
                </div>
                <div class="header-text">
                    <h1 class="header-title">รายงานการจัดส่ง</h1>
                    <p class="header-subtitle">ข้อมูลการจัดส่งพัสดุ แนวโน้ม และประสิทธิภาพการทำงาน</p>
                </div>
            </div>
            <div class="header-actions">
                <button onclick="exportReport()" class="btn btn-primary">
                    <i class="fas fa-download"></i>
                    <span>ส่งออกรายงาน</span>
                </button>
                <a href="{{ route('admin.reports.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span>กลับ</span>
                </a>
            </div>
        </div>
    </div>

    <!-- Filters Card -->
    <div class="filters-card">
        <div class="filters-header">
            <h3 class="filters-title">
                <i class="fas fa-filter"></i>
                ตัวกรองข้อมูล
            </h3>
            <button onclick="resetFilters()" class="btn-reset">
                <i class="fas fa-undo"></i>
                รีเซ็ต
            </button>
        </div>
        <form method="GET" action="{{ route('admin.reports.delivery') }}" class="filters-form">
            <div class="filters-grid">
                <div class="filter-group">
                    <label>วันที่เริ่มต้น</label>
                    <input type="date" name="start_date" value="{{ $startDate }}" class="filter-input">
                </div>
                <div class="filter-group">
                    <label>วันที่สิ้นสุด</label>
                    <input type="date" name="end_date" value="{{ $endDate }}" class="filter-input">
                </div>
                <div class="filter-group">
                    <label>คลังสินค้า</label>
                    <select name="warehouse" class="filter-select">
                        <option value="">ทั้งหมด</option>
                        @foreach($warehouses as $warehouse)
                            <option value="{{ $warehouse->code }}" {{ $warehouseCode == $warehouse->code ? 'selected' : '' }}>
                                {{ $warehouse->code }} - {{ $warehouse->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="filter-group">
                    <label>ประเภทรถ</label>
                    <select name="vehicle_type" class="filter-select">
                        <option value="">ทั้งหมด</option>
                        <option value="2wheel" {{ $vehicleType == '2wheel' ? 'selected' : '' }}>รถ 2 ล้อ</option>
                        <option value="4wheel" {{ $vehicleType == '4wheel' ? 'selected' : '' }}>รถ 4 ล้อ</option>
                    </select>
                </div>
                <div class="filter-group">
                    <label>พนักงาน</label>
                    <select name="user_id" class="filter-select">
                        <option value="">ทั้งหมด</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" {{ $userId == $user->id ? 'selected' : '' }}>
                                {{ $user->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="filter-actions">
                    <button type="submit" class="btn-filter">
                        <i class="fas fa-search"></i>
                        ค้นหา
                    </button>
                </div>
            </div>
        </form>
    </div>

    <!-- Stats Grid -->
    <div class="stats-grid">
        <div class="stat-card orange">
            <div class="stat-icon">
                <i class="fas fa-truck"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ number_format($stats['total_deliveries']) }}</div>
                <div class="stat-label">การจัดส่งทั้งหมด</div>
            </div>
        </div>

        <div class="stat-card blue">
            <div class="stat-icon">
                <i class="fas fa-boxes"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ number_format($stats['total_parcels']) }}</div>
                <div class="stat-label">จำนวนพัสดุ</div>
            </div>
        </div>

        <div class="stat-card green">
            <div class="stat-icon">
                <i class="fas fa-coins"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">฿{{ number_format($stats['total_commission'], 2) }}</div>
                <div class="stat-label">ค่าคอมมิชชั่นรวม</div>
            </div>
        </div>

        <div class="stat-card purple">
            <div class="stat-icon">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value">{{ number_format($stats['avg_parcels_per_delivery'], 1) }}</div>
                <div class="stat-label">เฉลี่ยพัสดุต่อครั้ง</div>
            </div>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="charts-section">
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-chart-area"></i>
                    แนวโน้มการจัดส่งรายวัน
                </h3>
            </div>
            <div class="chart-container">
                <canvas id="dailyTrendChart"></canvas>
            </div>
        </div>

        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-chart-pie"></i>
                    การกระจายตามคลังสินค้า
                </h3>
            </div>
            <div class="chart-container">
                <canvas id="warehouseChart"></canvas>
            </div>
        </div>
    </div>

    <!-- Top Performers -->
    <div class="section-card">
        <div class="section-header">
            <div class="section-icon orange">
                <i class="fas fa-trophy"></i>
            </div>
            <div class="section-info">
                <h2 class="section-title">พนักงานยอดเยี่ยม</h2>
                <p class="section-subtitle">10 อันดับแรกที่มีผลงานดีที่สุด</p>
            </div>
        </div>

        <div class="performers-grid">
            @foreach($topPerformers->take(10) as $index => $performer)
                <div class="performer-card {{ $index < 3 ? 'top-' . ($index + 1) : '' }}">
                    <div class="performer-rank">
                        @if($index < 3)
                            <i class="fas fa-medal"></i>
                        @endif
                        #{{ $index + 1 }}
                    </div>
                    <div class="performer-info">
                        <h4 class="performer-name">{{ $performer->user->name ?? 'N/A' }}</h4>
                        <p class="performer-warehouse">{{ $performer->user->warehouse_code ?? 'N/A' }}</p>
                    </div>
                    <div class="performer-stats">
                        <div class="performer-stat">
                            <span class="stat-number">{{ number_format($performer->total_parcels) }}</span>
                            <span class="stat-text">พัสดุ</span>
                        </div>
                        <div class="performer-stat">
                            <span class="stat-number">฿{{ number_format($performer->total_commission, 0) }}</span>
                            <span class="stat-text">คอมมิชชั่น</span>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>

    <!-- Data Table -->
    <div class="table-card">
        <div class="table-header">
            <h3 class="table-title">
                <i class="fas fa-table"></i>
                รายละเอียดการจัดส่ง
            </h3>
            <div class="table-actions">
                <div class="table-info">
                    แสดง {{ $deliveries->firstItem() ?? 0 }} - {{ $deliveries->lastItem() ?? 0 }} 
                    จาก {{ number_format($deliveries->total()) }} รายการ
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>วันที่</th>
                        <th>พนักงาน</th>
                        <th>คลังสินค้า</th>
                        <th>ประเภทรถ</th>
                        <th>จำนวนพัสดุ</th>
                        <th>ค่าคอมมิชชั่น</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($deliveries as $delivery)
                        <tr>
                            <td>{{ $delivery->delivery_date->format('d/m/Y') }}</td>
                            <td>
                                <div class="user-info">
                                    <span class="user-name">{{ $delivery->user->name ?? 'N/A' }}</span>
                                    <span class="user-email">{{ $delivery->user->email ?? '' }}</span>
                                </div>
                            </td>
                            <td>
                                <span class="warehouse-badge">{{ $delivery->user->warehouse_code ?? 'N/A' }}</span>
                            </td>
                            <td>
                                <span class="vehicle-badge {{ $delivery->user->role ?? '' }}">
                                    {{ $delivery->user->role == '2wheel' ? 'รถ 2 ล้อ' : 'รถ 4 ล้อ' }}
                                </span>
                            </td>
                            <td>
                                <span class="parcel-count">{{ number_format($delivery->parcel_count) }}</span>
                            </td>
                            <td>
                                <span class="commission-amount">฿{{ number_format($delivery->commission_amount, 2) }}</span>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="no-data">
                                <i class="fas fa-inbox"></i>
                                ไม่พบข้อมูลการจัดส่ง
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        @if($deliveries->hasPages())
            <div class="pagination-container">
                {{ $deliveries->appends(request()->query())->links() }}
            </div>
        @endif
    </div>
</div>
@endsection

@push('styles')
<style>
/* Container */
.delivery-reports-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 50%, #cbd5e1 100%);
    padding: 2rem;
}

/* Header Card */
.header-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 24px;
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
    margin-bottom: 2rem;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.header-info {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.header-icon {
    width: 64px;
    height: 64px;
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
    box-shadow: 0 10px 25px rgba(255, 107, 53, 0.3);
}

.header-title {
    font-size: 2rem;
    font-weight: 700;
    background: linear-gradient(135deg, #1f2937 0%, #6b7280 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    margin: 0 0 0.5rem 0;
}

.header-subtitle {
    color: #6b7280;
    margin: 0;
    font-size: 1rem;
}

.header-actions {
    display: flex;
    gap: 0.75rem;
}

/* Buttons */
.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 500;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
    color: white;
    box-shadow: 0 10px 25px rgba(255, 107, 53, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 15px 35px rgba(255, 107, 53, 0.4);
}

.btn-secondary {
    background: rgba(255, 255, 255, 0.9);
    color: #374151;
    border: 1px solid #e5e7eb;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.btn-secondary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
}

/* Filters Card */
.filters-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
    margin-bottom: 2rem;
}

.filters-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
}

.filters-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.btn-reset {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: #f3f4f6;
    color: #6b7280;
    border: none;
    border-radius: 8px;
    font-size: 0.875rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-reset:hover {
    background: #e5e7eb;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.filter-group label {
    font-weight: 500;
    color: #374151;
    font-size: 0.875rem;
}

.filter-input, .filter-select {
    padding: 0.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 0.875rem;
    transition: border-color 0.3s ease;
}

.filter-input:focus, .filter-select:focus {
    outline: none;
    border-color: #FF6B35;
}

.btn-filter {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
    border: none;
    border-radius: 8px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-filter:hover {
    transform: translateY(-1px);
    box-shadow: 0 10px 25px rgba(59, 130, 246, 0.3);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 16px;
    padding: 2rem;
    transition: all 0.3s ease;
    border: 1px solid rgba(255, 255, 255, 0.2);
    position: relative;
    overflow: hidden;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
}

.stat-card.orange {
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
    color: white;
}

.stat-card.blue {
    background: linear-gradient(135deg, #3B82F6 0%, #1D4ED8 100%);
    color: white;
}

.stat-card.green {
    background: linear-gradient(135deg, #10B981 0%, #059669 100%);
    color: white;
}

.stat-card.purple {
    background: linear-gradient(135deg, #8B5CF6 0%, #7C3AED 100%);
    color: white;
}

.stat-icon {
    width: 60px;
    height: 60px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    margin-bottom: 1rem;
}

.stat-value {
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 0.875rem;
    opacity: 0.9;
}

/* Charts Section */
.charts-section {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
}

.chart-header {
    margin-bottom: 1.5rem;
}

.chart-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.chart-container {
    height: 300px;
    position: relative;
}

/* Section Cards */
.section-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 24px;
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 2rem;
    margin-bottom: 2rem;
}

.section-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 2rem;
}

.section-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: white;
}

.section-icon.orange {
    background: linear-gradient(135deg, #FF6B35 0%, #DC2626 100%);
}

.section-title {
    font-size: 1.5rem;
    font-weight: 600;
    color: #1f2937;
    margin: 0;
}

.section-subtitle {
    color: #6b7280;
    font-size: 0.9rem;
    margin: 0.25rem 0 0 0;
}

/* Performers Grid */
.performers-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
}

.performer-card {
    background: #f8f9fa;
    border-radius: 12px;
    padding: 1.5rem;
    border: 2px solid transparent;
    transition: all 0.3s ease;
}

.performer-card:hover {
    background: white;
    border-color: #FF6B35;
    transform: translateY(-2px);
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
}

.performer-card.top-1 {
    background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
    color: white;
}

.performer-card.top-2 {
    background: linear-gradient(135deg, #C0C0C0 0%, #A9A9A9 100%);
    color: white;
}

.performer-card.top-3 {
    background: linear-gradient(135deg, #CD7F32 0%, #B8860B 100%);
    color: white;
}

.performer-rank {
    font-size: 1.5rem;
    font-weight: 700;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.performer-name {
    font-size: 1.1rem;
    font-weight: 600;
    margin: 0 0 0.25rem 0;
}

.performer-warehouse {
    font-size: 0.875rem;
    opacity: 0.8;
    margin: 0;
}

.performer-stats {
    display: flex;
    gap: 1rem;
    margin-top: 1rem;
}

.performer-stat {
    display: flex;
    flex-direction: column;
    align-items: center;
}

.stat-number {
    font-size: 1.25rem;
    font-weight: 700;
}

.stat-text {
    font-size: 0.75rem;
    opacity: 0.8;
}

/* Table Card */
.table-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    overflow: hidden;
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 2rem 2rem 1rem 2rem;
}

.table-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.table-info {
    color: #6b7280;
    font-size: 0.875rem;
}

.table-container {
    overflow-x: auto;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
}

.data-table th {
    background: #f8f9fa;
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.data-table td {
    padding: 1rem;
    border-bottom: 1px solid #f3f4f6;
}

.data-table tr:hover {
    background: #f8f9fa;
}

.user-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.user-name {
    font-weight: 500;
    color: #1f2937;
}

.user-email {
    font-size: 0.875rem;
    color: #6b7280;
}

.warehouse-badge {
    display: inline-block;
    padding: 0.25rem 0.75rem;
    background: #e0f2fe;
    color: #0277bd;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 500;
}

.vehicle-badge {
    display: inline-block;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 500;
}

.vehicle-badge.2wheel {
    background: #f3e8ff;
    color: #7c3aed;
}

.vehicle-badge.4wheel {
    background: #ecfdf5;
    color: #059669;
}

.parcel-count {
    font-weight: 600;
    color: #1f2937;
}

.commission-amount {
    font-weight: 600;
    color: #059669;
}

.no-data {
    text-align: center;
    padding: 3rem;
    color: #6b7280;
}

.no-data i {
    font-size: 3rem;
    margin-bottom: 1rem;
    display: block;
}

.pagination-container {
    padding: 1.5rem 2rem;
    border-top: 1px solid #f3f4f6;
}

/* Responsive */
@media (max-width: 768px) {
    .delivery-reports-container {
        padding: 1rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .header-actions {
        width: 100%;
        justify-content: stretch;
    }
    
    .btn {
        flex: 1;
        justify-content: center;
    }
    
    .filters-grid {
        grid-template-columns: 1fr;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .charts-section {
        grid-template-columns: 1fr;
    }
    
    .performers-grid {
        grid-template-columns: 1fr;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Daily Trend Chart
const dailyTrendCtx = document.getElementById('dailyTrendChart').getContext('2d');
const dailyTrendChart = new Chart(dailyTrendCtx, {
    type: 'line',
    data: {
        labels: {!! json_encode($dailyTrend->pluck('date')->map(function($date) { return date('d/m', strtotime($date)); })) !!},
        datasets: [{
            label: 'การจัดส่ง',
            data: {!! json_encode($dailyTrend->pluck('delivery_count')) !!},
            borderColor: '#FF6B35',
            backgroundColor: 'rgba(255, 107, 53, 0.1)',
            tension: 0.4,
            fill: true
        }, {
            label: 'จำนวนพัสดุ',
            data: {!! json_encode($dailyTrend->pluck('parcel_count')) !!},
            borderColor: '#3B82F6',
            backgroundColor: 'rgba(59, 130, 246, 0.1)',
            tension: 0.4,
            fill: true,
            yAxisID: 'y1'
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                type: 'linear',
                display: true,
                position: 'left',
            },
            y1: {
                type: 'linear',
                display: true,
                position: 'right',
                grid: {
                    drawOnChartArea: false,
                },
            }
        }
    }
});

// Warehouse Distribution Chart
const warehouseCtx = document.getElementById('warehouseChart').getContext('2d');
const warehouseChart = new Chart(warehouseCtx, {
    type: 'doughnut',
    data: {
        labels: {!! json_encode($warehouseDistribution->pluck('warehouse_code')) !!},
        datasets: [{
            data: {!! json_encode($warehouseDistribution->pluck('total_parcels')) !!},
            backgroundColor: [
                '#FF6B35',
                '#3B82F6',
                '#10B981',
                '#8B5CF6',
                '#F59E0B',
                '#EF4444',
                '#06B6D4',
                '#84CC16'
            ]
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom',
            }
        }
    }
});

function exportReport() {
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '{{ route("admin.reports.delivery.export") }}';
    
    // Add CSRF token
    const csrfToken = document.createElement('input');
    csrfToken.type = 'hidden';
    csrfToken.name = '_token';
    csrfToken.value = '{{ csrf_token() }}';
    form.appendChild(csrfToken);
    
    // Add current filters
    const params = new URLSearchParams(window.location.search);
    params.forEach((value, key) => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = key;
        input.value = value;
        form.appendChild(input);
    });
    
    document.body.appendChild(form);
    form.submit();
    document.body.removeChild(form);
}

function resetFilters() {
    window.location.href = '{{ route("admin.reports.delivery") }}';
}
</script>
@endpush