@extends('layouts.app')

@section('title', 'เพิ่มอัตราค่าคอมมิชชั่น - EazyCom Plus')
@section('page-title', 'เพิ่มอัตราค่าคอมมิชชั่น')
@section('mobile-title', 'เพิ่มอัตราคอมมิชชั่น')

@section('content')
<div class="create-commission-page">
    <!-- Header Section -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-info">
                <div class="header-title">
                    <div class="title-icon">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <div class="title-text">
                        <h1>เพิ่มอัตราค่าคอมมิชชั่น</h1>
                        <p>กำหนดอัตราค่าคอมมิชชั่นใหม่สำหรับคลังและประเภทรถ</p>
                    </div>
                </div>
                <div class="header-actions">
                    <a href="{{ route('admin.commission-rates.index') }}" class="action-btn secondary-btn">
                        <i class="fas fa-arrow-left"></i>
                        <span>กลับ</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Form Container -->
    <div class="form-container">
        <form method="POST" action="{{ route('admin.commission-rates.store') }}" id="commissionForm" class="commission-form">
            @csrf
            
            <!-- Basic Information Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-info-circle"></i>
                        <span>ข้อมูลพื้นฐาน</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="warehouse_code" class="form-label required">คลังสินค้า</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-warehouse"></i>
                                </div>
                                <select id="warehouse_code" name="warehouse_code" 
                                        class="form-select @error('warehouse_code') error @enderror" required>
                                    <option value="">เลือกคลังสินค้า</option>
                                    @foreach($warehouses ?? [] as $warehouse)
                                        <option value="{{ $warehouse->code }}" 
                                                {{ old('warehouse_code', $selectedWarehouse ?? '') === $warehouse->code ? 'selected' : '' }}>
                                            {{ $warehouse->code }} - {{ $warehouse->name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            @error('warehouse_code')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="vehicle_type" class="form-label required">ประเภทรถ</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-truck"></i>
                                </div>
                                <select id="vehicle_type" name="vehicle_type" 
                                        class="form-select @error('vehicle_type') error @enderror" required>
                                    <option value="">เลือกประเภทรถ</option>
                                    <option value="2wheel" {{ old('vehicle_type') === '2wheel' ? 'selected' : '' }}>
                                        🏍️ รถ 2 ล้อ
                                    </option>
                                    <option value="4wheel" {{ old('vehicle_type') === '4wheel' ? 'selected' : '' }}>
                                        🚚 รถ 4 ล้อ
                                    </option>
                                </select>
                            </div>
                            @error('vehicle_type')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tier Configuration Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-layer-group"></i>
                        <span>การกำหนดขั้น</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="tier" class="form-label required">ขั้น</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-hashtag"></i>
                                </div>
                                <input type="number" id="tier" name="tier" value="{{ old('tier') }}" 
                                       class="form-input @error('tier') error @enderror" 
                                       min="1" placeholder="เช่น 1, 2, 3" required>
                            </div>
                            @error('tier')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <div class="form-hint">ขั้นต้องไม่ซ้ำกันสำหรับคลังและประเภทรถเดียวกัน</div>
                        </div>

                        <div class="form-group">
                            <label for="rate_per_package" class="form-label required">อัตราค่าคอมมิชชั่นต่อชิ้น (บาท)</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-coins"></i>
                                </div>
                                <input type="number" id="rate_per_package" name="rate_per_package" value="{{ old('rate_per_package') }}" 
                                       class="form-input @error('rate_per_package') error @enderror" 
                                       step="0.01" min="0" placeholder="เช่น 2.50, 3.00" required>
                            </div>
                            @error('rate_per_package')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- Package Range Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-boxes"></i>
                        <span>ช่วงจำนวนพัสดุ</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="min_packages" class="form-label required">จำนวนพัสดุขั้นต่ำ</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-box"></i>
                                </div>
                                <input type="number" id="min_packages" name="min_packages" value="{{ old('min_packages') }}" 
                                       class="form-input @error('min_packages') error @enderror" 
                                       min="0" placeholder="เช่น 1, 50, 100" required>
                            </div>
                            @error('min_packages')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="max_packages" class="form-label">จำนวนพัสดุสูงสุด</label>
                            <div class="input-wrapper">
                                <div class="input-icon">
                                    <i class="fas fa-boxes"></i>
                                </div>
                                <input type="number" id="max_packages" name="max_packages" value="{{ old('max_packages') }}" 
                                       class="form-input @error('max_packages') error @enderror" 
                                       min="1" placeholder="เว้นว่างหากไม่จำกัด">
                            </div>
                            @error('max_packages')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <div class="form-hint">เว้นว่างไว้หากเป็นขั้นสุดท้าย (ไม่จำกัดจำนวนสูงสุด)</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Status Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-toggle-on"></i>
                        <span>สถานะการใช้งาน</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="status-options">
                        <label class="status-option active">
                            <input type="radio" name="is_active" value="1" 
                                   {{ old('is_active', '1') === '1' ? 'checked' : '' }}>
                            <div class="option-content">
                                <div class="option-icon active">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <div class="option-text">
                                    <div class="option-title">ใช้งานอยู่</div>
                                    <div class="option-desc">อัตรานี้จะถูกใช้ในการคำนวณ</div>
                                </div>
                            </div>
                        </label>
                        
                        <label class="status-option">
                            <input type="radio" name="is_active" value="0" 
                                   {{ old('is_active') === '0' ? 'checked' : '' }}>
                            <div class="option-content">
                                <div class="option-icon inactive">
                                    <i class="fas fa-times-circle"></i>
                                </div>
                                <div class="option-text">
                                    <div class="option-title">ปิดใช้งาน</div>
                                    <div class="option-desc">อัตรานี้จะไม่ถูกใช้ในการคำนวณ</div>
                                </div>
                            </div>
                        </label>
                    </div>
                </div>
            </div>

            <!-- Preview Card -->
            <div class="form-card">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-calculator"></i>
                        <span>ตัวอย่างการคำนวณ</span>
                    </div>
                </div>
                <div class="card-body">
                    <div id="calculation-preview" class="calculation-preview">
                        <div class="preview-placeholder">
                            <i class="fas fa-info-circle"></i>
                            <span>กรอกข้อมูลเพื่อดูตัวอย่างการคำนวณ</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Existing Rates Card -->
            <div id="existing-rates-card" class="form-card" style="display: none;">
                <div class="card-header">
                    <div class="card-title">
                        <i class="fas fa-list-alt"></i>
                        <span>อัตราที่มีอยู่แล้ว</span>
                    </div>
                </div>
                <div class="card-body">
                    <div id="existing-rates-content" class="existing-rates-list">
                        <!-- จะแสดงผ่าน JavaScript -->
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="form-actions">
                <div class="progress-indicator">
                    <div class="progress-step active">
                        <div class="step-icon">
                            <i class="fas fa-info-circle"></i>
                        </div>
                        <div class="step-text">ข้อมูลพื้นฐาน</div>
                    </div>
                    <div class="progress-line"></div>
                    <div class="progress-step">
                        <div class="step-icon">
                            <i class="fas fa-calculator"></i>
                        </div>
                        <div class="step-text">ตรวจสอบ</div>
                    </div>
                    <div class="progress-line"></div>
                    <div class="progress-step">
                        <div class="step-icon">
                            <i class="fas fa-check"></i>
                        </div>
                        <div class="step-text">เสร็จสิ้น</div>
                    </div>
                </div>
                
                <div class="action-buttons">
                    <a href="{{ route('admin.commission-rates.index') }}" class="btn btn-secondary">
                        <i class="fas fa-times"></i>
                        <span>ยกเลิก</span>
                    </a>
                    <button type="submit" class="btn btn-primary" id="submit-btn">
                        <i class="fas fa-save"></i>
                        <span>บันทึกอัตรา</span>
                        <div class="btn-loader" style="display: none;">
                            <i class="fas fa-spinner fa-spin"></i>
                        </div>
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<style>
/* Create Commission Page Styles */
.create-commission-page {
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

/* Page Header */
.page-header {
    background: white;
    border-bottom: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-bottom: 2rem;
}

.header-content {
    max-width: 1000px;
    margin: 0 auto;
    padding: 2rem;
}

.header-info {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 2rem;
}

.header-title {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.title-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 24px;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.title-text h1 {
    font-size: 2rem;
    font-weight: 600;
    color: #1e293b;
    margin: 0;
}

.title-text p {
    color: #64748b;
    margin: 0.5rem 0 0;
    font-size: 1rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.secondary-btn {
    background: #f1f5f9;
    color: #475569;
    border: 1px solid #e2e8f0;
}

.secondary-btn:hover {
    background: #e2e8f0;
    transform: translateY(-1px);
}

/* Form Container */
.form-container {
    max-width: 1000px;
    margin: 0 auto;
    padding: 0 2rem 2rem;
}

.commission-form {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

/* Form Cards */
.form-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
    border: 1px solid #e2e8f0;
    overflow: hidden;
    transition: all 0.3s ease;
}

.form-card:hover {
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
}

.card-header {
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.05) 0%, rgba(255, 140, 97, 0.05) 100%);
    padding: 1.5rem 2rem;
    border-bottom: 1px solid #e2e8f0;
}

.card-title {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.125rem;
    font-weight: 600;
    color: #1e293b;
}

.card-title i {
    color: #FF6B35;
    font-size: 1.25rem;
}

.card-body {
    padding: 2rem;
}

/* Form Grid */
.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-label {
    font-weight: 500;
    color: #374151;
    font-size: 0.95rem;
}

.form-label.required::after {
    content: '*';
    color: #ef4444;
    margin-left: 0.25rem;
}

/* Input Wrapper */
.input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.input-icon {
    position: absolute;
    left: 1rem;
    color: #9ca3af;
    font-size: 0.9rem;
    z-index: 10;
}

.form-input,
.form-select {
    width: 100%;
    padding: 0.875rem 1rem 0.875rem 2.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    font-size: 0.95rem;
    transition: all 0.3s ease;
    background: white;
    font-family: inherit;
}

.form-input:focus,
.form-select:focus {
    outline: none;
    border-color: #FF6B35;
    box-shadow: 0 0 0 3px rgba(255, 107, 53, 0.1);
}

.form-input.error,
.form-select.error {
    border-color: #ef4444;
    box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.1);
}

/* Status Options */
.status-options {
    display: flex;
    gap: 1rem;
}

.status-option {
    flex: 1;
    cursor: pointer;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    padding: 1.5rem;
    transition: all 0.3s ease;
    background: white;
}

.status-option:hover {
    border-color: #FF6B35;
    background: rgba(255, 107, 53, 0.02);
}

.status-option input[type="radio"] {
    display: none;
}

.status-option input[type="radio"]:checked + .option-content {
    color: #FF6B35;
}

.status-option.active {
    border-color: #FF6B35;
    background: rgba(255, 107, 53, 0.05);
}

.option-content {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.option-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: white;
}

.option-icon.active {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
}

.option-icon.inactive {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
}

.option-text {
    flex: 1;
}

.option-title {
    font-weight: 600;
    color: #1e293b;
    margin-bottom: 0.25rem;
}

.option-desc {
    font-size: 0.85rem;
    color: #64748b;
}

/* Calculation Preview */
.calculation-preview {
    min-height: 120px;
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
}

.preview-placeholder {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.95rem;
}

.preview-content {
    width: 100%;
}

.preview-tier {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    margin-bottom: 1rem;
}

.tier-info {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.tier-badge {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-weight: 600;
    font-size: 0.85rem;
}

.tier-range {
    color: #64748b;
}

.tier-rate {
    font-weight: 600;
    color: #FF6B35;
}

.preview-example {
    background: rgba(255, 107, 53, 0.1);
    border: 1px solid rgba(255, 107, 53, 0.2);
    border-radius: 8px;
    padding: 1rem;
}

.example-title {
    font-weight: 600;
    color: #1e293b;
    margin-bottom: 0.5rem;
}

.example-calc {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.9rem;
}

.example-result {
    font-weight: 600;
    color: #FF6B35;
}

/* Existing Rates List */
.existing-rates-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.existing-rate-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem 1.5rem;
    background: #f8fafc;
    border: 1px solid #e2e8f0;
    border-radius: 12px;
    transition: all 0.3s ease;
}

.existing-rate-item:hover {
    border-color: #FF6B35;
    background: rgba(255, 107, 53, 0.02);
}

.existing-rate-info {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.existing-tier-badge {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-weight: 600;
    font-size: 0.85rem;
}

.existing-rate-details {
    color: #64748b;
    font-size: 0.9rem;
}

.existing-rate-status {
    font-size: 0.85rem;
    font-weight: 500;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
}

.existing-rate-status.active {
    background: rgba(34, 197, 94, 0.1);
    color: #16a34a;
}

.existing-rate-status.inactive {
    background: rgba(239, 68, 68, 0.1);
    color: #dc2626;
}

/* Error Messages */
.error-message {
    color: #ef4444;
    font-size: 0.85rem;
    margin-top: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.error-message::before {
    content: '⚠️';
    font-size: 0.8rem;
}

/* Form Hints */
.form-hint {
    color: #64748b;
    font-size: 0.85rem;
    margin-top: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.form-hint::before {
    content: 'ℹ️';
    font-size: 0.8rem;
}

/* Form Actions */
.form-actions {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
    border: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.progress-indicator {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.progress-step {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    opacity: 0.4;
    transition: opacity 0.3s ease;
}

.progress-step.active {
    opacity: 1;
}

.step-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #e5e7eb;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6b7280;
    transition: all 0.3s ease;
}

.progress-step.active .step-icon {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
}

.step-text {
    font-size: 0.85rem;
    font-weight: 500;
    color: #64748b;
}

.progress-step.active .step-text {
    color: #1e293b;
    font-weight: 600;
}

.progress-line {
    width: 40px;
    height: 2px;
    background: #e5e7eb;
}

.action-buttons {
    display: flex;
    gap: 1rem;
}

.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 0.875rem 2rem;
    border: none;
    border-radius: 12px;
    font-family: inherit;
    font-weight: 500;
    font-size: 0.95rem;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    position: relative;
    overflow: hidden;
}

.btn-primary {
    background: linear-gradient(135deg, #FF6B35 0%, #FF8C61 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(255, 107, 53, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 53, 0.4);
}

.btn-primary:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-secondary {
    background: #f1f5f9;
    color: #475569;
    border: 1px solid #e2e8f0;
}

.btn-secondary:hover {
    background: #e2e8f0;
    transform: translateY(-1px);
}

.btn-loader {
    position: absolute;
    inset: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    background: inherit;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 1.5rem 1rem;
    }
    
    .header-info {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .form-container {
        padding: 0 1rem 2rem;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .card-body {
        padding: 1.5rem;
    }
    
    .status-options {
        flex-direction: column;
    }
    
    .form-actions {
        flex-direction: column;
        gap: 1.5rem;
        align-items: stretch;
    }
    
    .progress-indicator {
        justify-content: center;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: center;
    }
    
    .btn {
        flex: 1;
        justify-content: center;
    }
    
    .title-text h1 {
        font-size: 1.5rem;
    }
    
    .existing-rate-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.75rem;
    }
}

/* Animation */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.form-card {
    animation: fadeIn 0.5s ease-out;
}

.form-card:nth-child(1) { animation-delay: 0.1s; }
.form-card:nth-child(2) { animation-delay: 0.2s; }
.form-card:nth-child(3) { animation-delay: 0.3s; }
.form-card:nth-child(4) { animation-delay: 0.4s; }
.form-card:nth-child(5) { animation-delay: 0.5s; }
.form-card:nth-child(6) { animation-delay: 0.6s; }
</style>

<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('commissionForm');
    const submitBtn = document.getElementById('submit-btn');

    // Form field references
    const warehouseSelect = document.getElementById('warehouse_code');
    const vehicleSelect = document.getElementById('vehicle_type');
    const tierInput = document.getElementById('tier');
    const minPackagesInput = document.getElementById('min_packages');
    const maxPackagesInput = document.getElementById('max_packages');
    const rateInput = document.getElementById('rate_per_package');

    // Update calculation preview
    function updateCalculationPreview() {
        const minPackages = parseInt(minPackagesInput.value) || 0;
        const maxPackages = parseInt(maxPackagesInput.value) || null;
        const rate = parseFloat(rateInput.value) || 0;
        const tier = parseInt(tierInput.value) || 0;

        const previewDiv = document.getElementById('calculation-preview');

        if (minPackages > 0 && rate >= 0 && tier > 0) {
            const rangeText = maxPackages ? `${minPackages} - ${maxPackages}` : `${minPackages}+`;
            const exampleCount = maxPackages ? Math.min(minPackages + 20, maxPackages) : minPackages + 20;
            const packagesInTier = Math.max(0, exampleCount - minPackages + 1);
            const exampleCommission = packagesInTier * rate;

            previewDiv.innerHTML = `
                <div class="preview-content">
                    <div class="preview-tier">
                        <div class="tier-info">
                            <div class="tier-badge">ขั้นที่ ${tier}</div>
                            <div class="tier-range">${rangeText} ชิ้น</div>
                        </div>
                        <div class="tier-rate">${rate.toFixed(2)} บาท/ชิ้น</div>
                    </div>
                    
                    <div class="preview-example">
                        <div class="example-title">ตัวอย่างการคำนวณ</div>
                        <div class="example-calc">
                            หากมีพัสดุ ${exampleCount} ชิ้น → พัสดุในขั้นนี้: ${packagesInTier} ชิ้น
                        </div>
                        <div class="example-calc">
                            ค่าคอมมิชชั่น: ${packagesInTier} × ${rate.toFixed(2)} = 
                            <span class="example-result">${exampleCommission.toFixed(2)} บาท</span>
                        </div>
                    </div>
                </div>
            `;
        } else {
            previewDiv.innerHTML = `
                <div class="preview-placeholder">
                    <i class="fas fa-info-circle"></i>
                    <span>กรอกข้อมูลเพื่อดูตัวอย่างการคำนวณ</span>
                </div>
            `;
        }
    }

    // Load existing rates
    function loadExistingRates() {
        const warehouseCode = warehouseSelect.value;
        const vehicleType = vehicleSelect.value;

        if (warehouseCode && vehicleType) {
            // Simulate API call - replace with actual endpoint
            const mockRates = [
                { tier: 1, min_packages: 1, max_packages: 50, rate_per_package: 2.00, is_active: true },
                { tier: 2, min_packages: 51, max_packages: 100, rate_per_package: 2.50, is_active: true },
                { tier: 3, min_packages: 101, max_packages: null, rate_per_package: 3.00, is_active: false }
            ];

            if (mockRates.length > 0) {
                let html = '';
                mockRates.forEach(rate => {
                    const rangeText = rate.max_packages ? 
                        `${rate.min_packages} - ${rate.max_packages}` : 
                        `${rate.min_packages}+`;
                    const statusClass = rate.is_active ? 'active' : 'inactive';
                    const statusText = rate.is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน';

                    html += `
                        <div class="existing-rate-item">
                            <div class="existing-rate-info">
                                <div class="existing-tier-badge">ขั้นที่ ${rate.tier}</div>
                                <div class="existing-rate-details">
                                    ${rangeText} ชิ้น = ${rate.rate_per_package.toFixed(2)} บาท/ชิ้น
                                </div>
                            </div>
                            <div class="existing-rate-status ${statusClass}">${statusText}</div>
                        </div>
                    `;
                });

                document.getElementById('existing-rates-content').innerHTML = html;
                document.getElementById('existing-rates-card').style.display = 'block';
            } else {
                document.getElementById('existing-rates-card').style.display = 'none';
            }
        } else {
            document.getElementById('existing-rates-card').style.display = 'none';
        }
    }

    // Event listeners
    [minPackagesInput, maxPackagesInput, rateInput, tierInput].forEach(input => {
        input.addEventListener('input', updateCalculationPreview);
    });

    [warehouseSelect, vehicleSelect].forEach(select => {
        select.addEventListener('change', loadExistingRates);
    });

    // Status option handling
    document.querySelectorAll('.status-option input[type="radio"]').forEach(radio => {
        radio.addEventListener('change', function() {
            document.querySelectorAll('.status-option').forEach(option => {
                option.classList.remove('active');
            });
            
            if (this.checked) {
                this.closest('.status-option').classList.add('active');
            }
        });
    });

    // Initialize active status option
    const checkedRadio = document.querySelector('.status-option input[type="radio"]:checked');
    if (checkedRadio) {
        checkedRadio.closest('.status-option').classList.add('active');
    }

    // Form validation
    function validateForm() {
        const warehouseCode = warehouseSelect.value;
        const vehicleType = vehicleSelect.value;
        const tier = parseInt(tierInput.value);
        const minPackages = parseInt(minPackagesInput.value);
        const maxPackages = parseInt(maxPackagesInput.value);
        const rate = parseFloat(rateInput.value);

        if (!warehouseCode) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ครบถ้วน',
                text: 'กรุณาเลือกคลังสินค้า',
                confirmButtonColor: '#FF6B35'
            });
            warehouseSelect.focus();
            return false;
        }

        if (!vehicleType) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ครบถ้วน',
                text: 'กรุณาเลือกประเภทรถ',
                confirmButtonColor: '#FF6B35'
            });
            vehicleSelect.focus();
            return false;
        }

        if (!tier || tier < 1) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ถูกต้อง',
                text: 'กรุณาระบุขั้นที่ถูกต้อง',
                confirmButtonColor: '#FF6B35'
            });
            tierInput.focus();
            return false;
        }

        if (isNaN(minPackages) || minPackages < 0) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ถูกต้อง',
                text: 'กรุณาระบุจำนวนพัสดุขั้นต่ำที่ถูกต้อง',
                confirmButtonColor: '#FF6B35'
            });
            minPackagesInput.focus();
            return false;
        }

        if (maxPackages && maxPackages <= minPackages) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ถูกต้อง',
                text: 'จำนวนพัสดุสูงสุดต้องมากกว่าจำนวนขั้นต่ำ',
                confirmButtonColor: '#FF6B35'
            });
            maxPackagesInput.focus();
            return false;
        }

        if (isNaN(rate) || rate < 0) {
            Swal.fire({
                icon: 'error',
                title: 'ข้อมูลไม่ถูกต้อง',
                text: 'กรุณาระบุอัตราค่าคอมมิชชั่นที่ถูกต้อง',
                confirmButtonColor: '#FF6B35'
            });
            rateInput.focus();
            return false;
        }

        return true;
    }

    // Form submission
    form.addEventListener('submit', function(e) {
        e.preventDefault();

        if (!validateForm()) {
            return;
        }

        Swal.fire({
            title: 'ยืนยันการบันทึก',
            text: 'คุณต้องการบันทึกอัตราค่าคอมมิชชั่นใหม่นี้หรือไม่?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#FF6B35',
            cancelButtonColor: '#6B7280',
            confirmButtonText: 'บันทึก',
            cancelButtonText: 'ยกเลิก'
        }).then((result) => {
            if (result.isConfirmed) {
                // Show loading state
                submitBtn.disabled = true;
                submitBtn.querySelector('span').style.display = 'none';
                submitBtn.querySelector('.btn-loader').style.display = 'flex';

                // Update progress
                document.querySelectorAll('.progress-step').forEach((step, index) => {
                    if (index <= 1) {
                        step.classList.add('active');
                    }
                });

                // Submit form after delay
                setTimeout(() => {
                    form.submit();
                }, 1500);
            }
        });
    });

    // Initialize
    updateCalculationPreview();
    loadExistingRates();

    // Show success/error messages from session
    @if(session('success'))
        Swal.fire({
            icon: 'success',
            title: 'สำเร็จ!',
            text: '{{ session('success') }}',
            confirmButtonColor: '#FF6B35'
        });
    @endif

    @if(session('error'))
        Swal.fire({
            icon: 'error',
            title: 'เกิดข้อผิดพลาด!',
            text: '{{ session('error') }}',
            confirmButtonColor: '#FF6B35'
        });
    @endif

    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        // Ctrl/Cmd + S to save
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
            e.preventDefault();
            if (validateForm()) {
                form.dispatchEvent(new Event('submit'));
            }
        }
        
        // Escape to cancel
        if (e.key === 'Escape') {
            window.location.href = '{{ route('admin.commission-rates.index') }}';
        }
    });
});
</script>
@endsection