/**
 * I18n JavaScript Helper
 * Provides client-side internationalization support
 */

class I18n {
    constructor() {
        this.locale = document.documentElement.lang || 'th';
        this.translations = {};
        this.loadTranslations();
    }

    /**
     * Load translations from server
     */
    async loadTranslations() {
        try {
            const response = await fetch(`/api/translations/${this.locale}`);
            if (response.ok) {
                this.translations = await response.json();
            }
        } catch (error) {
            console.warn('Failed to load translations:', error);
        }
    }

    /**
     * Translate a key
     */
    __(key, replacements = {}) {
        let translation = this.get(key);
        
        // Replace placeholders
        Object.keys(replacements).forEach(placeholder => {
            const regex = new RegExp(`:${placeholder}`, 'g');
            translation = translation.replace(regex, replacements[placeholder]);
        });
        
        return translation;
    }

    /**
     * Get translation by key
     */
    get(key, defaultValue = null) {
        const keys = key.split('.');
        let value = this.translations;
        
        for (const k of keys) {
            if (value && typeof value === 'object' && k in value) {
                value = value[k];
            } else {
                return defaultValue || key;
            }
        }
        
        return value || defaultValue || key;
    }

    /**
     * Format number according to locale
     */
    formatNumber(number, decimals = 0) {
        if (this.locale === 'th') {
            return new Intl.NumberFormat('th-TH', {
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(number);
        }
        
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(number);
    }

    /**
     * Format currency according to locale
     */
    formatCurrency(amount, decimals = 2) {
        if (this.locale === 'th') {
            return new Intl.NumberFormat('th-TH', {
                style: 'currency',
                currency: 'THB',
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(amount);
        }
        
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(amount);
    }

    /**
     * Format date according to locale
     */
    formatDate(date, options = {}) {
        const dateObj = date instanceof Date ? date : new Date(date);
        
        const defaultOptions = {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        };
        
        const formatOptions = { ...defaultOptions, ...options };
        
        if (this.locale === 'th') {
            return new Intl.DateTimeFormat('th-TH', formatOptions).format(dateObj);
        }
        
        return new Intl.DateTimeFormat('en-US', formatOptions).format(dateObj);
    }

    /**
     * Format datetime according to locale
     */
    formatDateTime(datetime, options = {}) {
        const dateObj = datetime instanceof Date ? datetime : new Date(datetime);
        
        const defaultOptions = {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        };
        
        const formatOptions = { ...defaultOptions, ...options };
        
        if (this.locale === 'th') {
            return new Intl.DateTimeFormat('th-TH', formatOptions).format(dateObj);
        }
        
        return new Intl.DateTimeFormat('en-US', formatOptions).format(dateObj);
    }

    /**
     * Get relative time (e.g., "2 hours ago")
     */
    formatRelativeTime(date) {
        const dateObj = date instanceof Date ? date : new Date(date);
        const now = new Date();
        const diffInSeconds = Math.floor((now - dateObj) / 1000);
        
        const rtf = new Intl.RelativeTimeFormat(this.locale, { numeric: 'auto' });
        
        if (diffInSeconds < 60) {
            return rtf.format(-diffInSeconds, 'second');
        } else if (diffInSeconds < 3600) {
            return rtf.format(-Math.floor(diffInSeconds / 60), 'minute');
        } else if (diffInSeconds < 86400) {
            return rtf.format(-Math.floor(diffInSeconds / 3600), 'hour');
        } else if (diffInSeconds < 2592000) {
            return rtf.format(-Math.floor(diffInSeconds / 86400), 'day');
        } else if (diffInSeconds < 31536000) {
            return rtf.format(-Math.floor(diffInSeconds / 2592000), 'month');
        } else {
            return rtf.format(-Math.floor(diffInSeconds / 31536000), 'year');
        }
    }

    /**
     * Switch language
     */
    async switchLanguage(locale) {
        try {
            const response = await fetch('/language/switch', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({ locale })
            });

            const data = await response.json();
            
            if (data.success) {
                // Update localStorage
                localStorage.setItem('locale', locale);
                
                // Update current locale
                this.locale = locale;
                document.documentElement.lang = locale;
                
                // Reload translations
                await this.loadTranslations();
                
                return data;
            } else {
                throw new Error(data.message);
            }
        } catch (error) {
            console.error('Language switch failed:', error);
            throw error;
        }
    }

    /**
     * Get current locale
     */
    getLocale() {
        return this.locale;
    }

    /**
     * Check if current locale is RTL
     */
    isRtl() {
        return false; // Both Thai and English are LTR
    }
}

// Initialize global i18n instance
window.i18n = new I18n();

// Global helper functions
window.__ = (key, replacements) => window.i18n.__(key, replacements);
window.formatNumber = (number, decimals) => window.i18n.formatNumber(number, decimals);
window.formatCurrency = (amount, decimals) => window.i18n.formatCurrency(amount, decimals);
window.formatDate = (date, options) => window.i18n.formatDate(date, options);
window.formatDateTime = (datetime, options) => window.i18n.formatDateTime(datetime, options);
window.formatRelativeTime = (date) => window.i18n.formatRelativeTime(date);

// Auto-update elements with data-i18n attribute
document.addEventListener('DOMContentLoaded', function() {
    const updateI18nElements = () => {
        document.querySelectorAll('[data-i18n]').forEach(element => {
            const key = element.getAttribute('data-i18n');
            const replacements = element.getAttribute('data-i18n-params');
            const params = replacements ? JSON.parse(replacements) : {};
            
            element.textContent = window.__(key, params);
        });
    };
    
    // Initial update
    updateI18nElements();
    
    // Update when language changes
    document.addEventListener('languageChanged', updateI18nElements);
});

// Store locale in localStorage on page load
document.addEventListener('DOMContentLoaded', function() {
    const currentLocale = document.documentElement.lang;
    if (currentLocale) {
        localStorage.setItem('locale', currentLocale);
    }
});