<?php

namespace App\Http\Controllers;

use App\Models\SavingGoal;
use Illuminate\Http\Request;
use Carbon\Carbon;

class SavingGoalController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        $goals = SavingGoal::forUser($user->id)
            ->orderByRaw("FIELD(status, 'active', 'completed', 'cancelled')")
            ->orderBy('target_date')
            ->get();

        $stats = SavingGoal::getSummaryStats($user->id);

        return view('finance.goals.index', compact('goals', 'stats'));
    }

    public function create()
    {
        return view('finance.goals.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'goal_name' => 'required|string|max:255',
            'target_amount' => 'required|numeric|min:1',
            'target_date' => 'nullable|date|after:today',
            'initial_amount' => 'nullable|numeric|min:0',
        ], [
            'goal_name.required' => 'กรุณาระบุชื่อเป้าหมาย',
            'target_amount.required' => 'กรุณาระบุจำนวนเงินเป้าหมาย',
            'target_amount.min' => 'จำนวนเงินต้องมากกว่า 0',
            'target_date.after' => 'วันที่เป้าหมายต้องเป็นวันในอนาคต',
        ]);

        $goal = SavingGoal::create([
            'user_id' => auth()->id(),
            'goal_name' => $request->goal_name,
            'target_amount' => $request->target_amount,
            'current_amount' => $request->initial_amount ?? 0,
            'target_date' => $request->target_date,
            'status' => 'active',
        ]);

        return redirect()->route('finance.goals.index')
            ->with('success', 'สร้างเป้าหมายการออมเรียบร้อยแล้ว');
    }

    public function show(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        // Get deposit history (from finance records)
        $deposits = \App\Models\FinanceRecord::forUser($goal->user_id)
            ->where('note', 'like', '%' . $goal->goal_name . '%')
            ->orderBy('record_date', 'desc')
            ->get();

        return view('finance.goals.show', compact('goal', 'deposits'));
    }

    public function edit(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        return view('finance.goals.edit', compact('goal'));
    }

    public function update(Request $request, SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'goal_name' => 'required|string|max:255',
            'target_amount' => 'required|numeric|min:1',
            'target_date' => 'nullable|date',
        ]);

        $goal->update($request->only(['goal_name', 'target_amount', 'target_date']));

        return redirect()->route('finance.goals.index')
            ->with('success', 'อัปเดตเป้าหมายเรียบร้อยแล้ว');
    }

    public function deposit(Request $request, SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:1',
            'note' => 'nullable|string|max:255',
        ]);

        // Update goal
        $goal->deposit($request->amount);

        // Create finance record
        \App\Models\FinanceRecord::create([
            'user_id' => auth()->id(),
            'record_date' => now(),
            'type' => 'expense',
            'category' => 'savings',
            'amount' => $request->amount,
            'note' => "ออมเงิน: {$goal->goal_name}" . ($request->note ? " - {$request->note}" : ''),
            'source' => 'manual',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'บันทึกการออมเงินเรียบร้อยแล้ว',
            'current_amount' => $goal->current_amount,
            'progress_percentage' => $goal->progress_percentage,
            'is_completed' => $goal->is_completed,
        ]);
    }

    public function withdraw(Request $request, SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $goal->current_amount,
            'note' => 'nullable|string|max:255',
        ]);

        // Update goal
        $goal->withdraw($request->amount);

        // Create finance record
        \App\Models\FinanceRecord::create([
            'user_id' => auth()->id(),
            'record_date' => now(),
            'type' => 'income',
            'category' => 'withdrawal',
            'amount' => $request->amount,
            'note' => "ถอนเงินจาก: {$goal->goal_name}" . ($request->note ? " - {$request->note}" : ''),
            'source' => 'manual',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'ถอนเงินเรียบร้อยแล้ว',
            'current_amount' => $goal->current_amount,
            'progress_percentage' => $goal->progress_percentage,
        ]);
    }

    public function complete(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $goal->complete();

        return back()->with('success', 'บรรลุเป้าหมายเรียบร้อยแล้ว!');
    }

    public function cancel(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $goal->cancel();

        return back()->with('success', 'ยกเลิกเป้าหมายเรียบร้อยแล้ว');
    }

    public function reactivate(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $goal->reactivate();

        return back()->with('success', 'เปิดใช้งานเป้าหมายอีกครั้งเรียบร้อยแล้ว');
    }

    public function destroy(SavingGoal $goal)
    {
        // Check ownership
        if ($goal->user_id !== auth()->id()) {
            abort(403);
        }

        $goal->delete();

        return redirect()->route('finance.goals.index')
            ->with('success', 'ลบเป้าหมายเรียบร้อยแล้ว');
    }
}