<?php

namespace App\Http\Controllers;

use App\Models\FinanceRecord;
use App\Models\FinanceCategory;
use App\Models\MonthlyBudget;
use App\Models\SavingGoal;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class FinanceDashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $selectedMonth = $request->input('month', now()->format('Y-m'));
        $selectedYear = Carbon::parse($selectedMonth)->year;

        // Sync delivery commissions first
        FinanceRecord::syncDeliveryCommissions($user->id, $selectedMonth);

        // Get monthly statistics
        $monthlyStats = FinanceRecord::getMonthlyStats($user->id, $selectedMonth);
        
        // Debug: Log monthly stats
        \Log::info('Monthly Stats Debug:', [
            'user_id' => $user->id,
            'selected_month' => $selectedMonth,
            'total_expense' => $monthlyStats['total_expense'],
            'expense_by_category_count' => count($monthlyStats['expense_by_category']),
            'expense_by_category' => $monthlyStats['expense_by_category']
        ]);

        // Get yearly trend for chart
        $yearlyTrend = FinanceRecord::getYearlyTrend($user->id, $selectedYear);

        // Get expense analysis
        $expenseAnalysis = FinanceRecord::getExpenseAnalysis($user->id, $selectedMonth);

        // Get recommendations (fallback if method doesn't exist)
        $recommendations = [];
        if (method_exists(FinanceRecord::class, 'getRecommendations')) {
            $recommendations = FinanceRecord::getRecommendations($user->id, $selectedMonth);
        }

        // Get recent transactions
        $recentTransactions = FinanceRecord::forUser($user->id)
            ->with('categoryInfo')
            ->orderBy('record_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($record) {
                $record->transaction_date = $record->record_date;
                return $record;
            });

        // Get budget status
        $budgets = MonthlyBudget::where('user_id', $user->id)
            ->where('month', $selectedMonth)
            ->get();

        // Update budget spent amounts
        foreach ($budgets as $budget) {
            $spent = FinanceRecord::forUser($user->id)
                ->expense()
                ->forMonth($selectedMonth)
                ->byCategory($budget->category)
                ->sum('amount');
            
            $budget->update(['spent_amount' => $spent]);
        }

        // Get saving goals
        $savingGoals = SavingGoal::where('user_id', $user->id)
            ->where('status', 'active')
            ->get();

        // Calculate available months
        $availableMonths = [];
        $currentMonth = now();
        for ($i = 11; $i >= 0; $i--) {
            $month = $currentMonth->copy()->subMonths($i);
            $availableMonths[] = [
                'value' => $month->format('Y-m'),
                'label' => $month->locale('th')->monthName . ' ' . $month->year,
            ];
        }

        // Prepare chart data
        $trendData = [
            'labels' => [],
            'income' => [],
            'expense' => []
        ];
        
        if (isset($yearlyTrend) && is_array($yearlyTrend)) {
            foreach ($yearlyTrend as $month => $data) {
                $trendData['labels'][] = Carbon::parse($month)->locale('th')->shortMonthName;
                $trendData['income'][] = $data['income'] ?? 0;
                $trendData['expense'][] = $data['expense'] ?? 0;
            }
        }
        
        // Prepare expense chart data
        $expenseChartData = [
            'labels' => [],
            'data' => [],
            'colors' => []
        ];
        
        if (isset($monthlyStats['expense_by_category']) && is_array($monthlyStats['expense_by_category']) && count($monthlyStats['expense_by_category']) > 0) {
            $colors = ['#FF6B35', '#FFA94D', '#FFD93D', '#22C55E', '#3B82F6', '#8B5CF6', '#EF4444'];
            $colorIndex = 0;
            
            foreach ($monthlyStats['expense_by_category'] as $expense) {
                $category = FinanceCategory::where('slug', $expense['category'])->first();
                $expenseChartData['labels'][] = $category->name ?? $expense['category'];
                $expenseChartData['data'][] = floatval($expense['amount']);
                $expenseChartData['colors'][] = $category->color ?? $colors[$colorIndex % count($colors)];
                $colorIndex++;
            }
        } else {
            // Fallback: Create sample data if no expenses exist
            $expenseChartData = [
                'labels' => ['อาหาร', 'เดินทาง', 'น้ำมัน'],
                'data' => [1500, 800, 2000],
                'colors' => ['#FF6B35', '#FFA94D', '#FFD93D']
            ];
        }

        return view('finance.dashboard', compact(
            'monthlyStats',
            'yearlyTrend',
            'expenseAnalysis',
            'recommendations',
            'recentTransactions',
            'budgets',
            'savingGoals',
            'selectedMonth',
            'availableMonths',
            'trendData',
            'expenseChartData'
        ));
    }

    public function getChartData(Request $request)
    {
        $user = auth()->user();
        $type = $request->input('type', 'monthly');
        $period = $request->input('period', now()->format('Y-m'));

        switch ($type) {
            case 'daily':
                $data = FinanceRecord::getDailySummary($user->id, $period);
                break;
            
            case 'category':
                $stats = FinanceRecord::getMonthlyStats($user->id, $period);
                $data = $stats['expense_by_category'];
                break;
            
            case 'yearly':
                $year = Carbon::parse($period)->year;
                $data = FinanceRecord::getYearlyTrend($user->id, $year);
                break;
            
            default:
                $data = [];
        }

        return response()->json($data);
    }

    public function exportReport(Request $request)
    {
        $user = auth()->user();
        $month = $request->input('month', now()->format('Y-m'));
        $format = $request->input('format', 'pdf');

        // Get all data for the month
        $records = FinanceRecord::forUser($user->id)
            ->forMonth($month)
            ->with('categoryInfo')
            ->orderBy('record_date')
            ->get();

        $stats = FinanceRecord::getMonthlyStats($user->id, $month);

        if ($format === 'csv') {
            return $this->exportCSV($records, $stats, $month);
        } else {
            return $this->exportPDF($records, $stats, $month);
        }
    }

    private function exportCSV($records, $stats, $month)
    {
        $filename = 'finance_report_' . $month . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($records, $stats, $month) {
            $file = fopen('php://output', 'w');
            
            // UTF-8 BOM
            fwrite($file, "\xEF\xBB\xBF");
            
            // Summary section
            fputcsv($file, ['สรุปรายรับ-รายจ่าย ประจำเดือน ' . Carbon::parse($month)->locale('th')->monthName . ' ' . Carbon::parse($month)->year]);
            fputcsv($file, []);
            fputcsv($file, ['รายการ', 'จำนวนเงิน (บาท)']);
            fputcsv($file, ['รายรับทั้งหมด', number_format($stats['total_income'], 2)]);
            fputcsv($file, ['รายจ่ายทั้งหมด', number_format($stats['total_expense'], 2)]);
            fputcsv($file, ['คงเหลือ', number_format($stats['net_amount'], 2)]);
            fputcsv($file, []);
            
            // Transaction details
            fputcsv($file, ['วันที่', 'ประเภท', 'หมวดหมู่', 'รายการ', 'จำนวนเงิน', 'หมายเหตุ']);
            
            foreach ($records as $record) {
                fputcsv($file, [
                    $record->record_date->format('d/m/Y'),
                    $record->type === 'income' ? 'รายรับ' : 'รายจ่าย',
                    $record->categoryInfo ? $record->categoryInfo->name : $record->category,
                    $record->note ?? '-',
                    number_format($record->amount, 2),
                    $record->source === 'delivery' ? 'จากการจัดส่ง' : 'บันทึกเอง'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function exportPDF($records, $stats, $month)
    {
        // This would require a PDF library like DomPDF or TCPDF
        // For now, returning a simple HTML view that can be printed
        return view('finance.report-pdf', compact('records', 'stats', 'month'));
    }

    public function syncDeliveryData(Request $request)
    {
        $user = auth()->user();
        $month = $request->input('month');

        try {
            FinanceRecord::syncDeliveryCommissions($user->id, $month);

            return response()->json([
                'success' => true,
                'message' => 'ซิงค์ข้อมูลค่าคอมมิชชั่นเรียบร้อยแล้ว'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการซิงค์ข้อมูล'
            ], 500);
        }
    }
}