<?php
// app/Http/Controllers/CommissionController.php
namespace App\Http\Controllers;

use App\Models\CommissionRate;
use App\Models\Warehouse;
use Illuminate\Http\Request;

class CommissionController extends Controller
{
    public function index()
    {
        $commissions = CommissionRate::with('warehouse')
                                   ->orderBy('warehouse_code')
                                   ->orderBy('vehicle_type')
                                   ->orderBy('tier')
                                   ->get()
                                   ->groupBy(['warehouse_code', 'vehicle_type']);
        
        $warehouses = Warehouse::active()->get();
        
        return view('admin.commissions.index', compact('commissions', 'warehouses'));
    }

    public function create()
    {
        $warehouses = Warehouse::active()->get();
        return view('admin.commissions.create', compact('warehouses'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'warehouse_code' => 'required|exists:warehouses,code',
            'vehicle_type' => 'required|in:2wheel,4wheel',
            'tier' => 'required|integer|min:1',
            'min_packages' => 'required|integer|min:1',
            'max_packages' => 'nullable|integer|gt:min_packages',
            'rate_per_package' => 'required|numeric|min:0',
        ]);

        // ตรวจสอบว่าไม่มี tier ซ้ำ
        $existingTier = CommissionRate::where('warehouse_code', $request->warehouse_code)
                                    ->where('vehicle_type', $request->vehicle_type)
                                    ->where('tier', $request->tier)
                                    ->exists();

        if ($existingTier) {
            return back()->withErrors(['tier' => 'ขั้นนี้มีอยู่แล้วสำหรับคลังและประเภทรถนี้']);
        }

        CommissionRate::create($request->all());

        return redirect()->route('admin.commissions.index')
                        ->with('success', 'เพิ่มอัตราค่าคอมมิชชั่นเรียบร้อยแล้ว');
    }

    public function edit(CommissionRate $commission)
    {
        $warehouses = Warehouse::active()->get();
        return view('admin.commissions.edit', compact('commission', 'warehouses'));
    }

    public function update(Request $request, CommissionRate $commission)
    {
        $request->validate([
            'warehouse_code' => 'required|exists:warehouses,code',
            'vehicle_type' => 'required|in:2wheel,4wheel',
            'tier' => 'required|integer|min:1',
            'min_packages' => 'required|integer|min:1',
            'max_packages' => 'nullable|integer|gt:min_packages',
            'rate_per_package' => 'required|numeric|min:0',
        ]);

        // ตรวจสอบว่าไม่มี tier ซ้ำ (ยกเว้นตัวเอง)
        $existingTier = CommissionRate::where('warehouse_code', $request->warehouse_code)
                                    ->where('vehicle_type', $request->vehicle_type)
                                    ->where('tier', $request->tier)
                                    ->where('id', '!=', $commission->id)
                                    ->exists();

        if ($existingTier) {
            return back()->withErrors(['tier' => 'ขั้นนี้มีอยู่แล้วสำหรับคลังและประเภทรถนี้']);
        }

        $commission->update($request->all());

        return redirect()->route('admin.commissions.index')
                        ->with('success', 'แก้ไขอัตราค่าคอมมิชชั่นเรียบร้อยแล้ว');
    }

    public function destroy(CommissionRate $commission)
    {
        $commission->delete();

        return redirect()->route('admin.commissions.index')
                        ->with('success', 'ลบอัตราค่าคอมมิชชั่นเรียบร้อยแล้ว');
    }

    public function toggle(CommissionRate $commission)
    {
        $commission->update(['is_active' => !$commission->is_active]);

        return back()->with('success', 'เปลี่ยนสถานะอัตราค่าคอมมิชชั่นเรียบร้อยแล้ว');
    }

    // API สำหรับคำนวณค่าคอมมิชชั่น
    public function calculate(Request $request)
    {
        $request->validate([
            'warehouse_code' => 'required|exists:warehouses,code',
            'vehicle_type' => 'required|in:2wheel,4wheel',
            'package_count' => 'required|integer|min:1',
        ]);

        $result = CommissionRate::calculateCommission(
            $request->warehouse_code,
            $request->vehicle_type,
            $request->package_count
        );

        return response()->json($result);
    }
}