<?php
// app/Http/Controllers/Admin/UserManagementController.php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;


class UserManagementController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'verified', 'role:admin']);
    }

    /**
     * Display a listing of users
     */
    public function index(Request $request)
    {
        $query = User::with('warehouse');

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('warehouse_code', 'like', "%{$search}%");
            });
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->where('role', $request->role);
        }

        // Filter by status
        if ($request->filled('status')) {
            $isActive = $request->status === 'active';
            $query->where('is_active', $isActive);
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(20);

        // Statistics
        $allUsers = User::all();
        $stats = [
            'total' => $allUsers->count(),
            'active' => $allUsers->where('is_active', true)->count(),
            'inactive' => $allUsers->where('is_active', false)->count(),
            'admin' => $allUsers->where('role', 'admin')->count(),
            '2wheel' => $allUsers->where('role', '2wheel')->count(),
            '4wheel' => $allUsers->where('role', '4wheel')->count(),
        ];

        return view('admin.users.index', compact('users', 'stats'));
    }

    /**
     * Show the form for creating a new user
     */
    public function create()
    {
        $warehouses = Warehouse::active()->orderBy('code')->get();
        return view('admin.users.create', compact('warehouses'));
    }

    /**
     * Store a newly created user
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,2wheel,4wheel',
            'warehouse_code' => [
                'nullable',
                'string',
                'exists:warehouses,code',
                function ($attribute, $value, $fail) use ($request) {
                    if (in_array($request->role, ['2wheel', '4wheel']) && empty($value)) {
                        $fail('กรุณาเลือกคลังสินค้าสำหรับผู้ใช้ประเภทรถ');
                    }
                },
            ],
            'is_active' => 'required|boolean',
        ], [
            'first_name.required' => 'กรุณาระบุชื่อ',
            'last_name.required' => 'กรุณาระบุนามสกุล',
            'email.required' => 'กรุณาระบุอีเมล',
            'email.unique' => 'อีเมลนี้ถูกใช้งานแล้ว',
            'password.required' => 'กรุณาระบุรหัสผ่าน',
            'password.min' => 'รหัสผ่านต้องมีอย่างน้อย 8 ตัวอักษร',
            'password.confirmed' => 'การยืนยันรหัสผ่านไม่ตรงกัน',
            'role.required' => 'กรุณาเลือกบทบาท',
            'warehouse_code.exists' => 'คลังสินค้าที่เลือกไม่ถูกต้อง',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $userData = [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role' => $request->role,
                'is_active' => $request->boolean('is_active', true),
                'email_verified_at' => $request->boolean('auto_verify', false) ? now() : null,
            ];

            // Only add warehouse_code if it's a vehicle user
            if (in_array($request->role, ['2wheel', '4wheel'])) {
                $userData['warehouse_code'] = $request->warehouse_code;
            }

            User::create($userData);

            return redirect()
                ->route('admin.users.index')
                ->with('success', 'เพิ่มผู้ใช้ใหม่เรียบร้อยแล้ว');

        } catch (\Exception $e) {
            return back()
                ->withErrors(['error' => 'เกิดข้อผิดพลาดในการสร้างผู้ใช้: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Show the form for editing the specified user
     */
    public function edit(User $user)
    {
        $warehouses = Warehouse::active()->orderBy('code')->get();
        return view('admin.users.edit', compact('user', 'warehouses'));
    }

    /**
     * Update the specified user
     */
    public function update(Request $request, User $user)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'role' => [
                'required',
                'in:admin,2wheel,4wheel',
                function ($attribute, $value, $fail) use ($user) {
                    // Prevent changing own role
                    if ($user->id === auth()->id() && $value !== $user->role) {
                        $fail('คุณไม่สามารถเปลี่ยนบทบาทของตัวเองได้');
                    }
                },
            ],
            'warehouse_code' => [
                'nullable',
                'string',
                'exists:warehouses,code',
                function ($attribute, $value, $fail) use ($request) {
                    if (in_array($request->role, ['2wheel', '4wheel']) && empty($value)) {
                        $fail('กรุณาเลือกคลังสินค้าสำหรับผู้ใช้ประเภทรถ');
                    }
                },
            ],
            'is_active' => [
                'required',
                'boolean',
                function ($attribute, $value, $fail) use ($user) {
                    // Prevent deactivating own account
                    if ($user->id === auth()->id() && !$value) {
                        $fail('คุณไม่สามารถปิดการใช้งานตัวเองได้');
                    }
                },
            ],
        ], [
            'first_name.required' => 'กรุณาระบุชื่อ',
            'last_name.required' => 'กรุณาระบุนามสกุล',
            'email.required' => 'กรุณาระบุอีเมล',
            'email.unique' => 'อีเมลนี้ถูกใช้งานแล้ว',
            'password.min' => 'รหัสผ่านต้องมีอย่างน้อย 8 ตัวอักษร',
            'password.confirmed' => 'การยืนยันรหัสผ่านไม่ตรงกัน',
            'role.required' => 'กรุณาเลือกบทบาท',
            'warehouse_code.exists' => 'คลังสินค้าที่เลือกไม่ถูกต้อง',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $updateData = [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'role' => $request->role,
                'is_active' => $request->boolean('is_active'),
            ];

            // Update password if provided
            if ($request->filled('password')) {
                $updateData['password'] = Hash::make($request->password);
            }

            // Handle warehouse_code based on role
            if (in_array($request->role, ['2wheel', '4wheel'])) {
                $updateData['warehouse_code'] = $request->warehouse_code;
            } else {
                $updateData['warehouse_code'] = null;
            }

            $user->update($updateData);

            return redirect()
                ->route('admin.users.index')
                ->with('success', 'อัปเดตข้อมูลผู้ใช้เรียบร้อยแล้ว');

        } catch (\Exception $e) {
            return back()
                ->withErrors(['error' => 'เกิดข้อผิดพลาดในการอัปเดต: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Remove the specified user
     */
    public function destroy(User $user)
    {
        // Prevent deleting own account
        if ($user->id === auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'คุณไม่สามารถลบบัญชีตัวเองได้'
            ]);
        }

        // Check if user has delivery records
        if ($user->deliveryRecords()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'ไม่สามารถลบผู้ใช้ที่มีประวัติการจัดส่งได้'
            ]);
        }

        try {
            $user->delete();

            return response()->json([
                'success' => true,
                'message' => 'ลบผู้ใช้เรียบร้อยแล้ว'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการลบ: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Reset user password
     */
    public function resetPassword(User $user)
    {
        // Prevent resetting own password
        if ($user->id === auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'คุณไม่สามารถรีเซ็ตรหัสผ่านตัวเองได้'
            ]);
        }

        try {
            // Generate secure random password
            $newPassword = Str::random(12);
            
            $user->update([
                'password' => Hash::make($newPassword)
            ]);

            return response()->json([
                'success' => true,
                'message' => htmlspecialchars('รีเซ็ตรหัสผ่านเรียบร้อยแล้ว', ENT_QUOTES, 'UTF-8'),
                'new_password' => htmlspecialchars($newPassword, ENT_QUOTES, 'UTF-8')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการรีเซ็ตรหัสผ่าน: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Bulk operations for users
     */
    public function bulk(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:activate,deactivate,delete',
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'ข้อมูลไม่ถูกต้อง'
            ]);
        }

        $userIds = $request->user_ids;
        $action = $request->action;

        // Prevent operations on own account
        if (in_array(auth()->id(), $userIds)) {
            return response()->json([
                'success' => false,
                'message' => 'ไม่สามารถดำเนินการกับบัญชีตัวเองได้'
            ]);
        }

        try {
            $count = 0;

            switch ($action) {
                case 'activate':
                    $count = User::whereIn('id', $userIds)->update(['is_active' => true]);
                    break;

                case 'deactivate':
                    $count = User::whereIn('id', $userIds)->update(['is_active' => false]);
                    break;

                case 'delete':
                    // Check for users with delivery records
                    $usersWithRecords = User::whereIn('id', $userIds)
                        ->whereHas('deliveryRecords')
                        ->count();
                    
                    if ($usersWithRecords > 0) {
                        return response()->json([
                            'success' => false,
                            'message' => 'ไม่สามารถลบผู้ใช้ที่มีประวัติการจัดส่งได้'
                        ]);
                    }

                    $count = User::whereIn('id', $userIds)->delete();
                    break;
            }

            $actionText = [
                'activate' => 'เปิดใช้งาน',
                'deactivate' => 'ปิดใช้งาน',
                'delete' => 'ลบ'
            ];

            return response()->json([
                'success' => true,
                'message' => "{$actionText[$action]} {$count} รายการเรียบร้อยแล้ว",
                'affected' => $count
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาด: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Toggle user status (for compatibility with AdminDashboardController)
     */
    public function toggleUserStatus(User $user)
    {
        // Prevent toggling own status
        if ($user->id === auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'คุณไม่สามารถเปลี่ยนสถานะตัวเองได้'
            ]);
        }

        try {
            $user->update([
                'is_active' => !$user->is_active
            ]);

            $status = $user->is_active ? 'เปิดใช้งาน' : 'ปิดใช้งาน';

            return response()->json([
                'success' => true,
                'message' => "เปลี่ยนสถานะผู้ใช้เป็น {$status} เรียบร้อยแล้ว"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการเปลี่ยนสถานะ: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export users to CSV
     */
    public function export(Request $request)
    {
        try {
            $query = User::with('warehouse');

            // Apply same filters as index
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                      ->orWhere('last_name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('warehouse_code', 'like', "%{$search}%");
                });
            }

            if ($request->filled('role')) {
                $query->where('role', $request->role);
            }

            if ($request->filled('status')) {
                $isActive = $request->status === 'active';
                $query->where('is_active', $isActive);
            }

            $users = $query->orderBy('created_at', 'desc')->get();

            $filename = 'users_' . date('Y-m-d_H-i-s') . '.csv';
            
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ];

            $callback = function() use ($users) {
                $file = fopen('php://output', 'w');
                
                // UTF-8 BOM for proper Thai character display in Excel
                fwrite($file, "\xEF\xBB\xBF");
                
                // Headers
                fputcsv($file, [
                    'ID',
                    'ชื่อ',
                    'นามสกุล',
                    'อีเมล',
                    'บทบาท',
                    'รหัสคลัง',
                    'ชื่อคลัง',
                    'สถานะ',
                    'สร้างเมื่อ',
                    'อัปเดตเมื่อ'
                ]);

                // Data rows
                foreach ($users as $user) {
                    fputcsv($file, [
                        $user->id,
                        $user->first_name,
                        $user->last_name,
                        $user->email,
                        $user->role,
                        $user->warehouse_code ?? '',
                        $user->warehouse ? $user->warehouse->name : '',
                        $user->is_active ? 'ใช้งานอยู่' : 'ปิดใช้งาน',
                        $user->created_at ? $user->created_at->format('d/m/Y H:i:s') : '',
                        $user->updated_at ? $user->updated_at->format('d/m/Y H:i:s') : '',
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);

        } catch (\Exception $e) {
            return back()->with('error', 'เกิดข้อผิดพลาดในการส่งออกข้อมูล: ' . $e->getMessage());
        }
    }
}