<?php

namespace App\Console\Commands;

use App\Models\Vehicle;
use App\Services\VehicleNotificationService;
use Illuminate\Console\Command;

class SendVehicleReports extends Command
{
    protected $signature = 'vehicle:send-reports {--type=daily}';
    protected $description = 'ส่งรายงานให้กับ vehicle users';

    public function handle(VehicleNotificationService $notificationService)
    {
        $type = $this->option('type');
        $vehicles = Vehicle::with('user')->whereHas('user')->get();

        foreach ($vehicles as $vehicle) {
            $reportData = $this->generateReportData($vehicle, $type);
            $period = $this->getPeriodText($type);
            
            $notificationService->sendReportNotification(
                $vehicle,
                "รายงาน{$type}",
                $reportData,
                $period
            );
        }

        $this->info("ส่งรายงาน {$type} ให้ {$vehicles->count()} คันเรียบร้อย");
    }

    private function generateReportData($vehicle, $type)
    {
        $query = $vehicle->deliveryRecords();
        
        switch ($type) {
            case 'daily':
                $query->whereDate('delivery_date', today());
                break;
            case 'weekly':
                $query->whereBetween('delivery_date', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'monthly':
                $query->whereMonth('delivery_date', now()->month);
                break;
        }

        $records = $query->get();
        
        return [
            'deliveries' => $records->count(),
            'distance' => $records->sum('distance'),
            'commission' => $records->sum('commission_amount'),
            'fuel_cost' => $records->sum('fuel_cost') ?? 0
        ];
    }

    private function getPeriodText($type)
    {
        return match($type) {
            'daily' => 'วันที่ ' . today()->format('d/m/Y'),
            'weekly' => 'สัปดาห์ที่ ' . now()->weekOfYear . ' ปี ' . now()->year,
            'monthly' => 'เดือน ' . now()->format('m/Y'),
            default => $type
        };
    }
}