@extends('layouts.app')

@section('title', 'รายงานการจัดส่ง - ระบบจัดการค่าคอมมิชชั่น')

@section('content')
<div class="content-container">
    <!-- Page Header -->
    <div class="page-header">
        <div class="page-header-content">
            <div class="page-header-text">
                <h1 class="page-title">
                    <i class="fas fa-truck text-yellow-500"></i>
                    <span>รายงานการจัดส่ง</span>
                </h1>
                <p class="page-subtitle">วิเคราะห์ข้อมูลการจัดส่งพัสดุและประสิทธิภาพการทำงาน</p>
            </div>
            <div class="page-header-actions">
                <a href="{{ route('admin.reports.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span class="btn-text">กลับ</span>
                </a>
                <button onclick="exportReport()" class="btn btn-primary">
                    <i class="fas fa-download"></i>
                    <span class="btn-text">ส่งออกรายงาน</span>
                </button>
            </div>
        </div>
    </div>

    <!-- Filter Section -->
    <div class="filter-card">
        <div class="filter-header">
            <h3 class="filter-title">
                <i class="fas fa-filter text-blue-500"></i>
                ตัวกรองข้อมูล
            </h3>
        </div>
        <form method="GET" action="{{ route('admin.reports.delivery') }}" class="filter-form">
            <div class="filter-grid">
                <div class="form-group">
                    <label class="form-label">วันที่เริ่มต้น</label>
                    <input type="date" name="start_date" value="{{ $startDate }}" 
                           class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">วันที่สิ้นสุด</label>
                    <input type="date" name="end_date" value="{{ $endDate }}" 
                           class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">คลังสินค้า</label>
                    <select name="warehouse" class="form-control">
                        <option value="">ทั้งหมด</option>
                        @foreach($warehouses as $warehouse)
                            <option value="{{ $warehouse->code }}" 
                                    {{ $warehouseCode == $warehouse->code ? 'selected' : '' }}>
                                {{ $warehouse->code }} - {{ $warehouse->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">ประเภทรถ</label>
                    <select name="vehicle_type" class="form-control">
                        <option value="">ทั้งหมด</option>
                        <option value="2wheel" {{ $vehicleType == '2wheel' ? 'selected' : '' }}>
                            🏍️ รถ 2 ล้อ
                        </option>
                        <option value="4wheel" {{ $vehicleType == '4wheel' ? 'selected' : '' }}>
                            🚚 รถ 4 ล้อ
                        </option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">พนักงาน</label>
                    <select name="user_id" class="form-control">
                        <option value="">ทั้งหมด</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" 
                                    {{ $userId == $user->id ? 'selected' : '' }}>
                                {{ $user->name }} ({{ $user->email }})
                            </option>
                        @endforeach
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">&nbsp;</label>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i>
                        <span>กรองข้อมูล</span>
                    </button>
                </div>
            </div>
        </form>
    </div>

    <!-- Summary Stats -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-content">
                <div class="stat-icon orange">
                    <i class="fas fa-truck"></i>
                </div>
                <div class="stat-info">
                    <div class="stat-value">{{ number_format($stats['total_deliveries']) }}</div>
                    <div class="stat-label">จำนวนการจัดส่ง</div>
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-content">
                <div class="stat-icon blue">
                    <i class="fas fa-boxes"></i>
                </div>
                <div class="stat-info">
                    <div class="stat-value">{{ number_format($stats['total_parcels']) }}</div>
                    <div class="stat-label">จำนวนพัสดุทั้งหมด</div>
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-content">
                <div class="stat-icon green">
                    <i class="fas fa-coins"></i>
                </div>
                <div class="stat-info">
                    <div class="stat-value">฿{{ number_format($stats['total_commission'], 2) }}</div>
                    <div class="stat-label">ค่าคอมมิชชั่นรวม</div>
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-content">
                <div class="stat-icon purple">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-info">
                    <div class="stat-value">{{ number_format($stats['avg_parcels_per_delivery'], 1) }}</div>
                    <div class="stat-label">เฉลี่ยพัสดุ/การส่ง</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="charts-grid">
        <!-- Daily Trend Chart -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-chart-area text-blue-500"></i>
                    แนวโน้มรายวัน
                </h3>
            </div>
            <div class="chart-body">
                <canvas id="dailyTrendChart"></canvas>
            </div>
        </div>

        <!-- Vehicle Type Distribution -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <i class="fas fa-chart-pie text-green-500"></i>
                    สัดส่วนตามประเภทรถ
                </h3>
            </div>
            <div class="chart-body">
                <canvas id="vehicleTypeChart"></canvas>
            </div>
        </div>
    </div>

    <!-- Top Performers -->
    <div class="data-card">
        <div class="data-header">
            <h3 class="data-title">
                <i class="fas fa-trophy text-yellow-500"></i>
                ผู้ส่งพัสดุสูงสุด 10 อันดับ
            </h3>
        </div>
        <div class="table-responsive">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>อันดับ</th>
                        <th>ชื่อพนักงาน</th>
                        <th>ประเภทรถ</th>
                        <th>คลังสินค้า</th>
                        <th>จำนวนครั้ง</th>
                        <th>จำนวนพัสดุ</th>
                        <th>ค่าคอมมิชชั่น</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($topPerformers as $index => $performer)
                    <tr>
                        <td class="text-center">
                            <span class="rank-badge rank-{{ $index + 1 }}">
                                {{ $index + 1 }}
                            </span>
                        </td>
                        <td>
                            <div class="user-info">
                                <div class="user-name">{{ $performer->user->name ?? 'N/A' }}</div>
                                <div class="user-email">{{ $performer->user->email ?? '' }}</div>
                            </div>
                        </td>
                        <td>
                            <span class="vehicle-badge {{ $performer->user->role ?? '' }}">
                                @if(($performer->user->role ?? '') == '2wheel')
                                    <i class="fas fa-motorcycle"></i> 2 ล้อ
                                @else
                                    <i class="fas fa-truck"></i> 4 ล้อ
                                @endif
                            </span>
                        </td>
                        <td>{{ $performer->user->warehouse_code ?? '-' }}</td>
                        <td class="text-center">{{ number_format($performer->delivery_count) }}</td>
                        <td class="text-center">{{ number_format($performer->total_parcels) }}</td>
                        <td class="text-right">฿{{ number_format($performer->total_commission, 2) }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>

    <!-- Warehouse Distribution -->
    <div class="data-card">
        <div class="data-header">
            <h3 class="data-title">
                <i class="fas fa-warehouse text-purple-500"></i>
                การกระจายตามคลังสินค้า
            </h3>
        </div>
        <div class="table-responsive">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>คลังสินค้า</th>
                        <th>จำนวนพนักงาน</th>
                        <th>จำนวนการส่ง</th>
                        <th>จำนวนพัสดุ</th>
                        <th>ค่าคอมมิชชั่นรวม</th>
                        <th>เฉลี่ยต่อคน</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($warehouseDistribution as $warehouse)
                    <tr>
                        <td>
                            <div class="warehouse-info">
                                <div class="warehouse-code">{{ $warehouse->warehouse_code }}</div>
                                <div class="warehouse-name">{{ $warehouse->user->warehouse->name ?? '' }}</div>
                            </div>
                        </td>
                        <td class="text-center">{{ number_format($warehouse->user_count) }}</td>
                        <td class="text-center">{{ number_format($warehouse->delivery_count) }}</td>
                        <td class="text-center">{{ number_format($warehouse->total_parcels) }}</td>
                        <td class="text-right">฿{{ number_format($warehouse->total_commission, 2) }}</td>
                        <td class="text-right">
                            ฿{{ $warehouse->user_count > 0 ? number_format($warehouse->total_commission / $warehouse->user_count, 2) : '0.00' }}
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>

    <!-- Detailed Delivery Records -->
    <div class="data-card">
        <div class="data-header">
            <h3 class="data-title">
                <i class="fas fa-list text-orange-500"></i>
                รายละเอียดการจัดส่ง
            </h3>
        </div>
        <div class="table-responsive">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>วันที่</th>
                        <th>พนักงาน</th>
                        <th>ประเภทรถ</th>
                        <th>คลังสินค้า</th>
                        <th>จำนวนพัสดุ</th>
                        <th>ค่าคอมมิชชั่น</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($deliveries as $delivery)
                    <tr>
                        <td>{{ $delivery->delivery_date->format('d/m/Y') }}</td>
                        <td>
                            <div class="user-info">
                                <div class="user-name">{{ $delivery->user->name ?? 'N/A' }}</div>
                                <div class="user-email">{{ $delivery->user->email ?? '' }}</div>
                            </div>
                        </td>
                        <td>
                            <span class="vehicle-badge {{ $delivery->user->role ?? '' }}">
                                @if(($delivery->user->role ?? '') == '2wheel')
                                    <i class="fas fa-motorcycle"></i> 2 ล้อ
                                @else
                                    <i class="fas fa-truck"></i> 4 ล้อ
                                @endif
                            </span>
                        </td>
                        <td>{{ $delivery->user->warehouse_code ?? '-' }}</td>
                        <td class="text-center">{{ number_format($delivery->parcel_count) }}</td>
                        <td class="text-right">฿{{ number_format($delivery->commission_amount, 2) }}</td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="6" class="text-center text-gray-500 py-4">
                            ไม่พบข้อมูลการจัดส่ง
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        @if($deliveries->hasPages())
        <div class="table-pagination">
            {{ $deliveries->withQueryString()->links() }}
        </div>
        @endif
    </div>
</div>
@endsection

@push('styles')
<style>
/* Filter Card */
.filter-card {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
    margin-bottom: 2rem;
    overflow: hidden;
}

.filter-header {
    background: linear-gradient(135deg, rgba(59, 130, 246, 0.05) 0%, rgba(37, 99, 235, 0.05) 100%);
    padding: 1rem 1.5rem;
    border-bottom: 1px solid var(--medium-gray);
}

.filter-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--dark-gray);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.filter-form {
    padding: 1.5rem;
}

.filter-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    align-items: end;
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
}

.stat-content {
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}

.stat-icon.orange {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    color: #d97706;
}

.stat-icon.blue {
    background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
    color: #2563eb;
}

.stat-icon.green {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #059669;
}

.stat-icon.purple {
    background: linear-gradient(135deg, #ede9fe 0%, #ddd6fe 100%);
    color: #7c3aed;
}

.stat-value {
    font-size: 1.75rem;
    font-weight: 700;
    color: var(--dark-gray);
    margin-bottom: 0.25rem;
}

/* Charts */
.charts-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
}

.chart-header {
    padding: 1.5rem;
    border-bottom: 1px solid var(--medium-gray);
}

.chart-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--dark-gray);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.chart-body {
    padding: 1.5rem;
}

/* Data Tables */
.data-card {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--medium-gray);
    margin-bottom: 2rem;
    overflow: hidden;
}

.data-header {
    padding: 1.5rem;
    border-bottom: 1px solid var(--medium-gray);
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.02) 0%, rgba(255, 140, 97, 0.02) 100%);
}

.data-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--dark-gray);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
}

.data-table th {
    background: #f8f9fa;
    padding: 0.75rem 1rem;
    text-align: left;
    font-weight: 600;
    font-size: 0.875rem;
    color: var(--dark-gray);
    border-bottom: 2px solid var(--medium-gray);
}

.data-table td {
    padding: 0.75rem 1rem;
    border-bottom: 1px solid #f1f5f9;
}

/* Rank Badge */
.rank-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    font-weight: 600;
    font-size: 0.875rem;
}

.rank-badge.rank-1 {
    background: #ffd700;
    color: #854d0e;
}

.rank-badge.rank-2 {
    background: #c0c0c0;
    color: #374151;
}

.rank-badge.rank-3 {
    background: #cd7f32;
    color: white;
}

.rank-badge:not(.rank-1):not(.rank-2):not(.rank-3) {
    background: #e5e7eb;
    color: #374151;
}

/* Vehicle Badge */
.vehicle-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.25rem 0.75rem;
    border-radius: var(--radius-full);
    font-size: 0.8rem;
    font-weight: 500;
}

.vehicle-badge.2wheel {
    background: #dbeafe;
    color: #2563eb;
}

.vehicle-badge.4wheel {
    background: #d1fae5;
    color: #059669;
}

/* User Info */
.user-info {
    display: flex;
    flex-direction: column;
    gap: 0.125rem;
}

.user-name {
    font-weight: 500;
    color: var(--dark-gray);
}

.user-email {
    font-size: 0.8rem;
    color: var(--gray);
}

/* Warehouse Info */
.warehouse-info {
    display: flex;
    flex-direction: column;
    gap: 0.125rem;
}

.warehouse-code {
    font-weight: 600;
    color: var(--dark-gray);
}

.warehouse-name {
    font-size: 0.8rem;
    color: var(--gray);
}

/* Pagination */
.table-pagination {
    padding: 1rem;
    border-top: 1px solid var(--medium-gray);
    display: flex;
    justify-content: center;
}

/* Responsive */
@media (max-width: 1024px) {
    .charts-grid {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .filter-grid {
        grid-template-columns: 1fr;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .page-header-actions {
        flex-direction: column;
        width: 100%;
    }
    
    .page-header-actions .btn {
        width: 100%;
        justify-content: center;
    }
    
    .btn-text {
        display: none;
    }
    
    .table-responsive {
        overflow-x: auto;
    }
    
    .data-table {
        min-width: 600px;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
// Daily Trend Chart
const dailyTrendCtx = document.getElementById('dailyTrendChart').getContext('2d');
const dailyTrendChart = new Chart(dailyTrendCtx, {
    type: 'line',
    data: {
        labels: {!! json_encode($dailyTrend->pluck('date')->map(function($date) {
            return \Carbon\Carbon::parse($date)->format('d/m');
        })) !!},
        datasets: [{
            label: 'จำนวนพัสดุ',
            data: {!! json_encode($dailyTrend->pluck('parcel_count')) !!},
            borderColor: 'rgb(59, 130, 246)',
            backgroundColor: 'rgba(59, 130, 246, 0.1)',
            tension: 0.1,
            fill: true
        }, {
            label: 'ค่าคอมมิชชั่น',
            data: {!! json_encode($dailyTrend->pluck('commission_amount')) !!},
            borderColor: 'rgb(34, 197, 94)',
            backgroundColor: 'rgba(34, 197, 94, 0.1)',
            tension: 0.1,
            fill: true,
            yAxisID: 'y1'
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                position: 'left',
                title: {
                    display: true,
                    text: 'จำนวนพัสดุ'
                }
            },
            y1: {
                beginAtZero: true,
                position: 'right',
                title: {
                    display: true,
                    text: 'ค่าคอมมิชชั่น (บาท)'
                },
                grid: {
                    drawOnChartArea: false
                }
            }
        }
    }
});

// Vehicle Type Chart
const vehicleTypeCtx = document.getElementById('vehicleTypeChart').getContext('2d');
const vehicleTypeChart = new Chart(vehicleTypeCtx, {
    type: 'doughnut',
    data: {
        labels: {!! json_encode($vehicleTypeDistribution->map(function($item) {
            return $item->role == '2wheel' ? 'รถ 2 ล้อ' : 'รถ 4 ล้อ';
        })) !!},
        datasets: [{
            data: {!! json_encode($vehicleTypeDistribution->pluck('total_parcels')) !!},
            backgroundColor: [
                'rgba(59, 130, 246, 0.8)',
                'rgba(34, 197, 94, 0.8)'
            ],
            borderColor: [
                'rgb(59, 130, 246)',
                'rgb(34, 197, 94)'
            ],
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom',
            }
        }
    }
});

// Export Report Function
function exportReport() {
    Swal.fire({
        title: 'เลือกรูปแบบการส่งออก',
        showCancelButton: true,
        confirmButtonText: 'CSV',
        cancelButtonText: 'ยกเลิก',
        confirmButtonColor: '#22C55E',
        cancelButtonColor: '#6B7280',
    }).then((result) => {
        if (result.isConfirmed) {
            const params = new URLSearchParams(window.location.search);
            window.location.href = `{{ route('admin.reports.delivery.export') }}?${params.toString()}`;
        }
    });
}
</script>
@endpush