<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SavingTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'saving_goal_id',
        'amount',
        'type',
        'note',
        'balance_after',
        'transaction_date'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'transaction_date' => 'datetime'
    ];

    /**
     * Relationships
     */
    public function savingGoal(): BelongsTo
    {
        return $this->belongsTo(SavingGoal::class);
    }

    /**
     * Scopes
     */
    public function scopeDeposits($query)
    {
        return $query->where('type', 'deposit');
    }

    public function scopeWithdrawals($query)
    {
        return $query->where('type', 'withdrawal');
    }

    public function scopeAutoDeductions($query)
    {
        return $query->where('type', 'auto_deduct');
    }

    /**
     * Attributes
     */
    public function getFormattedAmountAttribute()
    {
        $sign = in_array($this->type, ['deposit', 'auto_deduct']) ? '+' : '-';
        return $sign . '฿' . number_format($this->amount, 2);
    }

    public function getTypeLabelAttribute()
    {
        return [
            'deposit' => 'ฝากเงิน',
            'withdrawal' => 'ถอนเงิน',
            'auto_deduct' => 'หักอัตโนมัติ'
        ][$this->type] ?? $this->type;
    }

    public function getTypeColorAttribute()
    {
        return [
            'deposit' => '#27AE60',
            'withdrawal' => '#E74C3C',
            'auto_deduct' => '#3498DB'
        ][$this->type] ?? '#6C757D';
    }

    public function getTypeIconAttribute()
    {
        return [
            'deposit' => 'fa-arrow-up',
            'withdrawal' => 'fa-arrow-down',
            'auto_deduct' => 'fa-sync'
        ][$this->type] ?? 'fa-circle';
    }
}